import Plugin from 'src/plugin-system/plugin.class';
import HttpClient from 'src/service/http-client.service';

/**
 * Plugin for handling product comparison functionality including adding/removing products
 * from comparison lists and managing related UI interactions.
 *
 * @extends Plugin
 */
export default class CompareProductsAddPlugin extends Plugin {

    /**
     * Default configuration options for the plugin
     * @type {Object}
     */
    static options = {
        addButtonSelector: '.topdata-compare-products-add',
        removeButtonSelector: '.topdata-compare-products-remove-from-list'
    };

    /**
     * Initialize the plugin by registering event listeners
     */
    init() {
        this._registerEventListeners();
    }

    // ---- Event Listener Management ----

    /**
     * Register click event listeners for add/remove buttons
     * Removes existing listeners before adding new ones to prevent duplicates
     */
    _registerEventListeners() {
        if (this._onClickHandler) {
            this.el.removeEventListener('click', this._onClickHandler);
        }

        this._onClickHandler = (event) => {
            const addButton = event.target.closest(this.options.addButtonSelector);
            const removeButton = event.target.closest(this.options.removeButtonSelector);

            if (addButton) {
                event.preventDefault();
                this.onAdd(addButton);
                return;
            }

            if (removeButton) {
                event.preventDefault();
                this.onRemove(removeButton);
            }
        };

        this.el.addEventListener('click', this._onClickHandler);
    }

    // ---- Product Addition/Removal Handlers ----

    /**
     * Handle adding a product to the comparison list
     * @param {HTMLElement} buttonEl - The add button element that was clicked
     */
    onAdd(buttonEl) {
        const httpClient = new HttpClient(window.accessKey, window.contextToken);
        const path = buttonEl.dataset.path;

        if (typeof path !== 'string') {
            console.error('No data-path found on compare button.', buttonEl);
            this.getCompareProductsModalPlugin()?.$emitter.publish('openModalEvent', {plugin: this});
            document.querySelector('#topdata-compare-products-modal-trigger').click();
            return;
        }

        httpClient.get(path, (response) => {
            this.productCompareAdded(response)
        });
    }

    /**
     * Handle removing a product from the comparison list
     * @param {HTMLElement} buttonEl - The remove button element that was clicked
     */
    onRemove(buttonEl) {
        const httpClient = new HttpClient(window.accessKey, window.contextToken);
        const path = buttonEl.dataset.path;

        if (typeof path !== 'string') {
            console.error('No data-path found on compare remove button.', buttonEl);
            return;
        }

        httpClient.get(path, (response) => {
            this.productCompareRemoved(response);
        });
    }

    // ---- Response Processing ----

    /**
     * Process the response after a product has been removed from comparison
     * Updates UI elements and displays appropriate modals
     * @param {string} data - JSON response string from the server
     */
    productCompareRemoved(data) {
        data = JSON.parse(data);
        if (data.success !== true) {
            this.modalShowByData(data.text, null, true, data.modalTitle);
            return;
        }
        document.querySelectorAll('.topdata-compare-products-count-total').forEach(el => {
            el.innerHTML = data.count;
        });

        const comparePathLink = document.querySelector('.topdata-compare-products-path');
        if (comparePathLink) {
            if(data.count > 0) {
                comparePathLink.classList.remove('topdata-compare-products-hidden');
            } else {
                comparePathLink.classList.add('topdata-compare-products-hidden');
            }
        }

        this.modalShowByData(data.text, data.productImage, true, data.modalTitle);

        document.querySelectorAll('.topdata-compare-products-product-comparing-' + data.productId).forEach(el => {
            el.classList.add('topdata-compare-products-hidden');
        });
        document.querySelectorAll('.topdata-compare-products-product-compare-' + data.productId).forEach(el => {
            el.classList.remove('topdata-compare-products-hidden');
        });
    }

    /**
     * Process the response after a product has been added to comparison
     * Updates UI elements, displays modals, and handles first product special case
     * @param {string} data - JSON response string from the server
     */
    productCompareAdded(data) {
        data = JSON.parse(data);
        if (data.success !== true) {
            this.modalShowByData(data.text, null, true, data.modalTitle);
            return;
        }
        document.querySelectorAll('.topdata-compare-products-count-total').forEach(el => {
            el.innerHTML = data.count;
        });

        const comparePathLink = document.querySelector('.topdata-compare-products-path');
        if (comparePathLink) {
            comparePathLink.classList.remove('topdata-compare-products-hidden');
        }

        if (data.count === 1) {
            this.popupRender();
        } else {
            this.modalShowByData(data.text, data.productImage, data.popup, data.modalTitle);
        }

        document.querySelectorAll('.topdata-compare-products-product-comparing-' + data.productId).forEach(el => {
            el.classList.remove('topdata-compare-products-hidden');
        });
        document.querySelectorAll('.topdata-compare-products-product-compare-' + data.productId).forEach(el => {
            el.classList.add('topdata-compare-products-hidden');
        });

        this.onAjaxContentEvents();
    }

    // ---- UI Helper Methods ----

    /**
     * Display a temporary hint popup for first product addition
     * Shows hint for 4 seconds then auto-hides
     */
    popupRender() {
        const hintEl = document.querySelector('.topdata-compare-products-hint');
        if (hintEl) {
            hintEl.classList.remove('topdata-compare-products-hint-hidden');
            setTimeout(() => {
                hintEl.classList.add('topdata-compare-products-hint-hidden');
            }, 4000);
        }
    }

    /**
     * Get the CompareProductsModalPlugin instance if available
     * @returns {Object|null} The modal plugin instance or null if not found
     */
    getCompareProductsModalPlugin() {
        const modalEl = document.querySelector('#topdata-compare-products-modal');
        if (modalEl) {
            return window.PluginManager.getPluginInstanceFromElement(modalEl, 'CompareProductsModalPlugin');
        }
        return null;
    }

    /**
     * Display a modal with product comparison data
     * Sets up modal content and event handlers for comparison actions
     * @param {string} text - The text content to display in the modal
     * @param {string|null} image - Optional product image URL
     * @param {boolean} openTrigger - Whether to trigger the modal opening
     * @param {string|null} title - Optional modal title
     */
    modalShowByData(text, image = null, openTrigger = true, title = null) {
        const modal = document.querySelector('#topdata-compare-added-modal');
        if (!modal) return;

        const titleEl = modal.querySelector('.modal-title');
        if (titleEl && title) {
            titleEl.innerHTML = title;
        }

        // Add debug logs for modal content
        console.log('%c[DEBUG] Modal content being set:', 'color: blue; font-weight: bold;', { text, image, title });
        modal.querySelector('.topdata-compare-products-product-info').innerHTML = text;
        const imgEl = modal.querySelector('.topdata-compare-products-product-image');

        if (image) {
            imgEl.src = image;
            imgEl.style.display = 'block';
        } else {
            imgEl.style.display = 'none';
        }

        // Check if "Show compare" button exists
        const showCompareBtn = modal.querySelector('.topdata-compare-products-show-compare');
        console.log('%c[DEBUG] "Show compare" button found:', 'color: blue; font-weight: bold;', showCompareBtn);

        if (showCompareBtn) {
            console.log('%c[DEBUG] "Show compare" button found', 'color: blue; font-weight: bold;');
            // Proper event delegation for the button
            showCompareBtn.addEventListener('click', (e) => {
                console.log('%c[DEBUG] "Show compare" button clicked!', 'color: green; font-weight: bold;');

                // Close the "Comparison List Full" modal
                const closeButton = modal.querySelector('.btn-close');
                if (closeButton) {
                    closeButton.click();
                }

                // Open the main comparison modal
                const modalPlugin = this.getCompareProductsModalPlugin();
                if (modalPlugin) {
                    modalPlugin.$emitter.publish('openModalEvent', {open: true});
                }
            });
        }

        if (openTrigger) {
            const trigger = document.querySelector('#topdata-compare-products-modal-added-trigger');
            if (trigger) {
                console.log('%c[DEBUG] Triggering modal display', 'color: blue;');
                trigger.click();
            }
        }
    }

    // ---- AJAX Content Event Handling ----

    /**
     * Handle events after AJAX content is loaded
     * Triggers group change event setup
     */
    onAjaxContentEvents() {
        this.onGroupChange();
    }

    /**
     * Set up event listeners for group selector elements
     * Binds click handlers to comparison group selection buttons
     */
    onGroupChange() {
        let groupSelectorElements = document.getElementsByClassName('topdata-compare-products-popup-compare-group');
        for (let i = 0; i < groupSelectorElements.length; i++) {
            groupSelectorElements[i].addEventListener('click', this._onChangeGroup.bind(this));
        }
    }

    /**
     * Handle group selection changes in the comparison interface
     * Opens the comparison modal with the selected group and closes current modal
     * @param {Event} event - The click event from the group selector
     */
    _onChangeGroup(event) {
        let group = event.currentTarget.dataset.group ?? null;

        const modalPlugin = this.getCompareProductsModalPlugin();
        if (modalPlugin) {
            modalPlugin.$emitter.publish('openModalEvent', {plugin: this, group: group, open: true});
        }

        const closeButton = document.querySelector('#topdata-compare-added-modal .btn-close');
        if (closeButton) {
            closeButton.click();
        }
    }
}