<?php declare(strict_types=1);

namespace Topdata\FreeTopdataCompareProducts\Controller;

use Shopware\Core\Framework\Routing\RoutingException;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\FreeTopdataCompareProducts\Component\Helper;
use Topdata\FreeTopdataCompareProducts\Storefront\Page\PopupCompareProductsWidget\PopupCompareProductsWidgetLoader;
use Shopware\Core\Framework\Uuid\Uuid;

/**
 * 08/2025 created (extracted from CompareController)
 */
#[Route(defaults: ["_routeScope" => ["storefront"]])]
class CompareApiController extends StorefrontController
{
    /**
     * @param PopupCompareProductsWidgetLoader $popupCompareProductsWidgetLoader
     */
    public function __construct(
        private readonly PopupCompareProductsWidgetLoader $popupCompareProductsWidgetLoader
    )
    {
    }

    /**
     * Removes a product from the compare list.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return JsonResponse
     */
    #[Route('/topdata-compare-products/product-remove/{productId}',
        name: 'frontend.topdata-compare-products.api.product-remove',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function productRemove(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $productId = $request->get('productId');

        // ---- Validate product ID
        if (!Helper::isValidUuid($productId)) {
            return new JsonResponse(['success' => false, 'text' => 'Product id invalid']);
        }

        // ---- Remove product by ID
        $result = $this->popupCompareProductsWidgetLoader->removeProductById($productId, $request, $salesChannelContext);

        // ---- Return result
        if ($result['success']) {
            return new JsonResponse($result);
        }

        return new JsonResponse(['success' => false, 'text' => 'Something went wrong']);
    }

    /**
     * Adds a product to the compare list.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return JsonResponse
     */
    #[Route('/topdata-compare-products/product-add/{productId}',
        name: 'frontend.topdata-compare-products.api.product-add',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function productAdd(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $productId = $request->get('productId');

        // ---- Validate product ID
        if (!Helper::isValidUuid($productId)) {
            return new JsonResponse(['success' => false, 'text' => 'Product id invalid']);
        }

        // ---- Add product by ID
        $result = $this->popupCompareProductsWidgetLoader->addProductById($productId, $request, $salesChannelContext);

        // ---- Return result
        if ($result) {
            return new JsonResponse($result);
        }

        return new JsonResponse(['success' => false, 'text' => 'Something went wrong']);
    }

    /**
     * Loads the compare products popup.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return JsonResponse
     */
    #[Route('/topdata-compare-products/load-popup',
        name: 'frontend.topdata-compare-products.api.load-popup',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function loadGroup(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $result = ['success' => true];
        $page = $this->popupCompareProductsWidgetLoader->load($request, $salesChannelContext);

        // ---- Render the popup content
        $body = $this->renderStorefront('@FreeTopdataCompareProducts/storefront/widget/compare-products/compare-popup-content.html.twig', [
            'page' => $page
        ])->getContent();

        // ---- Prepare the result
        $result['count'] = Helper::getCompareProductsTotalCount($request);
        $result['body'] = $body;
        $result['title'] = $this->trans('topdata-compare-products.compareProducts');
//        if($page->productArts) {
//            $result['title'] .= ' "' . $page->productArts[0] . '"';
//        }

        return new JsonResponse($result);
    }

    /**
     * Loads the state of the compare products list (product IDs and count).
     *
     */
    #[Route('/topdata-compare-products/load-state',
        name: 'frontend.topdata-compare-products.api.load-state',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function loadState(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $this->popupCompareProductsWidgetLoader->checkTranslation($request, $salesChannelContext);

        return new JsonResponse([
            'productIds'   => Helper::getProductCompareIds($request),
            'productCount' => Helper::getCompareProductsTotalCount($request)
        ]);
    }

}