<?php declare(strict_types=1);

namespace Topdata\FreeTopdataCompareProducts\Component;

use Shopware\Core\System\SystemConfig\SystemConfigService;

/**
 * Service class to retrieve and manage configuration settings for the FreeTopdataCompareProducts plugin.
 * This class provides methods to access configuration values of different types (bool, string, int)
 * and handles default values if a configuration key is not set.
 */
class SettingsService
{
    /**
     * @var array
     */
    private $config;

    public function __construct(SystemConfigService $systemConfigService)
    {
        $this->config = $systemConfigService->get('FreeTopdataCompareProducts.config');        
    }
    
    /**
     * Retrieves a boolean configuration value by its key.
     *
     * @param string $key The configuration key.
     * @return bool The boolean value of the configuration, or false if not set.
     */
    public function getBool(string $key) : bool
    {
        return isset($this->config[$key]) ? (bool)$this->config[$key] : false;
    }
    
    /**
     * Retrieves a string configuration value by its key.
     *
     * @param string $key The configuration key.
     * @return string The string value of the configuration, or an empty string if not set.
     */
    public function getString(string $key) : string
    {
        return isset($this->config[$key]) ? (string)$this->config[$key] : '';
    }
    
    /**
     * Retrieves an integer configuration value by its key.
     *
     * @param string $key The configuration key.
     * @param bool $notNegative If true, ensures the returned value is not negative.
     * @return int The integer value of the configuration, or 0 if not set.
     */
    public function getInt(string $key, bool $notNegative = false) : int
    {
        $ret = isset($this->config[$key]) ? (int)$this->config[$key] : 0;
        
        // ---- Ensure the returned value is not negative if requested
        if($notNegative && ($ret<0)) {
            $ret = 0;
        }
        
        return $ret;
    }
    
    /**
     * Returns the entire configuration array.
     *
     * @return array
     */
    public function getConfig()
    {
        return $this->config;
    }
    
    /**
     * Retrieves a configuration value by its key.
     *
     * @param string $key The configuration key.
     * @return mixed The value of the configuration, or null if not set.
     */
    public function getValue(string $key)
    {
        return isset($this->config[$key]) ? $this->config[$key] : null;
    }
}