<?php

namespace Topdata\TopdataColors\Service;

use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\System\SystemConfig\SystemConfigService;

/**
 * Service for handling color-related functionality.
 * TODO: move this service to TopdataFoundation and use it in the connector plugin
 */
class ColorService
{
    /**
     * Get default color mapping.
     */
    const DEFAULT_MAP_COLOR_NAME_TO_CSS_COLOR = [
        'transparent'             => '#fff linear-gradient(45deg,rgba(0,0,0,.15) 25%,transparent 25%,transparent 50%,rgba(0,0,0,.15) 50%,rgba(0,0,0,.15) 75%,transparent 75%,transparent)',
        'eisfarben'               => '#f0f8ff',
        'anthrazit'               => '#494D53',
        'antikweiß'               => '#faebd7',
        'wasser'                  => '#00ffff',
        'aquamarinblau'           => '#7fffd4',
        'himmelblau'              => '#f0ffff',
        'beige'                   => '#f5f5dc',
        'biskuit'                 => '#ffe4c4',
        'schwarz matt'            => '#000000',
        'schwarz'                 => '#000000',
        'photo schwarz'           => '#000000',
        'titan'                   => '#868686',
        'mandelweiß'              => '#ffebcd',
        'blau'                    => '#0000ff',
        'blauviolett'             => '#8a2be2',
        'braun'                   => '#a52a2a',
        'gelbbraun'               => '#deb887',
        'kadettenblau'            => '#5f9ea0',
        'hellgrün'                => '#7fff00',
        'schokolade'              => '#d2691e',
        'koralle'                 => '#ff7f50',
        'kornblumenblau'          => '#6495ed',
        'mais'                    => '#fff8dc',
        'karmesinrot'             => '#dc143c',
        'türkis'                  => '#00ffff',
        'dunkelblau'              => '#00008b',
        'dunkeltürkis'            => '#008b8b',
        'dunkle goldrutenfarbe'   => '#b8860b',
        'dunkelgrün'              => '#006400',
        'dunkelgrau'              => '#a9a9a9',
        'dunkelkhaki'             => '#bdb76b',
        'dunkelmagenta'           => '#8b008b',
        'dunkles olivgrün'        => '#556b2f',
        'dunkles orange'          => '#ff8c00',
        'dunkle orchidee'         => '#9932cc',
        'dunkelrot'               => '#8b0000',
        'dunkle lachsfarbe'       => '#e9967a',
        'dunkles seegrün'         => '#8fbc8f',
        'dunkles schieferblau'    => '#483d8b',
        'dunkles schiefergrau'    => '#2f4f4f',
        'dunkelvilolett'          => '#9400d3',
        'tiefrosa'                => '#ff1493',
        'mehrfarbig'              => 'linear-gradient(45deg, red , yellow, blue)',
        'tiefes himmelblau'       => '#00bfff',
        'persenningblau'          => '#1e90ff',
        'backstein'               => '#b22222',
        'blütenweiß'              => '#fffaf0',
        'waldgrün'                => '#228b22',
        'geisterweiß'             => '#f8f8ff',
        'goldrute'                => '#daa520',
        'grün'                    => '#008000',
        'grüngelb'                => '#adff2f',
        'rauchgrau'               => '#808000',
        'honigmelone'             => '#f0fff0',
        'leuchtendes rosa'        => '#ff69b4',
        'indischrot'              => '#cd5c5c',
        'elfenbein'               => '#fffff0',
        'staubfarben'             => '#f0e68c',
        'lavendel'                => '#e6e6fa',
        'graphit'                 => '#45484D',
        'lavendelrosa'            => '#fff0f5',
        'rasengrün'               => '#7cfc00',
        'chiffongelb'             => '#fffacd',
        'hellblau'                => '#add8e6',
        'helles korallenrot'      => '#f08080',
        'helles cyan'             => '#e0ffff',
        'helles magenta'          => '#FFCCFF',
        'helles goldrutengelb'    => '#fafad2',
        'hellrosa'                => '#ffb6c1',
        'helle lachsfarbe'        => '#ffa07a',
        'helles seegrün'          => '#20b2aa',
        'helles himmelblau'       => '#87cefa',
        'helles schiefergrau'     => '#778899',
        'helles stahlblau'        => '#b0c4de',
        'hellgelb'                => '#ffffe0',
        'limone'                  => '#00ff00',
        'limonengrün'             => '#32cd32',
        'leinen'                  => '#faf0e6',
        'kastanie'                => '#800000',
        'mittleres aquamarin'     => '#66cdaa',
        'mittleres blau'          => '#0000cd',
        'mittlere orchidee'       => '#ba55d3',
        'mittleres violett'       => '#9370db',
        'mittleres seegrün'       => '#3cb371',
        'mittleres schieferblau'  => '#7b68ee',
        'mittleres frühlingsgrün' => '#00fa9a',
        'mittlere türkis'         => '#48d1cc',
        'mittleres violettrot'    => '#c71585',
        'mitternachtsblau'        => '#191970',
        'cremige minze'           => '#f5fffa',
        'altrosa'                 => '#ffe4e1',
        'mokassin'                => '#ffe4b5',
        'navajoweiß'              => '#ffdead',
        'marinenblau'             => '#000080',
        'alte spitze'             => '#fdf5e6',
        'olivgrün'                => '#808000',
        'olivgraubraun'           => '#6b8e23',
        'orangerot'               => '#ff4500',
        'orchidee'                => '#da70d6',
        'blasse goldrutenfarbe'   => '#eee8aa',
        'blassgrün'               => '#98fb98',
        'blasstürkis'             => '#afeeee',
        'blasses violettrot'      => '#db7093',
        'papayacreme'             => '#ffefd5',
        'pfirsich'                => '#ffdab9',
        'rosa'                    => '#ffc0cb',
        'pflaume'                 => '#dda0dd',
        'taubenblau'              => '#b0e0e6',
        'violett'                 => '#800080',
        'rot'                     => '#ff0000',
        'rosiges braun'           => '#bc8f8f',
        'königsbalu'              => '#4169e1',
        'sattelbraun'             => '#8b4513',
        'lachsfarben'             => '#fa8072',
        'sandbraun'               => '#f4a460',
        'seegrün'                 => '#2e8b57',
        'muschel'                 => '#fff5ee',
        'siennaerde'              => '#a0522d',
        'silber'                  => '#c0c0c0',
        'schieferblau'            => '#6a5acd',
        'schiefergrau'            => '#708090',
        'schneeweiß'              => '#fffafa',
        'frühlingsgrün'           => '#00ff7f',
        'stahlblau'               => '#4682b4',
        'hautfarben'              => '#d2b48c',
        'krickentengrün'          => '#008080',
        'distel'                  => '#d8bfd8',
        'tomate'                  => '#ff6347',
        'veilchen'                => '#ee82ee',
        'weizen'                  => '#f5deb3',
        'weiß'                    => '#ffffff',
        'rauchfarben'             => '#f5f5f5',
        'gelb'                    => '#ffff00',
        'gelbgrün'                => '#9acd32',
        'aliceblue'               => '#f0f8ff',
        'antiquewhite'            => '#faebd7',
        'aqua'                    => '#00ffff',
        'aquamarine'              => '#7fffd4',
        'azure'                   => '#f0ffff',
        'bisque'                  => '#ffe4c4',
        'black foto'              => '#000000',
        'black'                   => '#000000',
        'blanchedal- mond'        => '#ffebcd',
        'blue'                    => '#0000ff',
        'blueviolet'              => '#8a2be2',
        'brown'                   => '#a52a2a',
        'burlywood'               => '#deb887',
        'cadetblue'               => '#5f9ea0',
        'chartreuse'              => '#7fff00',
        'chocolate'               => '#d2691e',
        'coral'                   => '#ff7f50',
        'cornflowerblue'          => '#6495ed',
        'cornsilk'                => '#fff8dc',
        'crimson'                 => '#dc143c',
        'cyan'                    => '#00ffff',
        'darkblue'                => '#00008b',
        'darkcyan'                => '#008b8b',
        'darkgoldenrod'           => '#b8860b',
        'darkgreen'               => '#006400',
        'darkgrey'                => '#a9a9a9',
        'darkkhaki'               => '#bdb76b',
        'darkmagenta'             => '#8b008b',
        'darkolivegreen'          => '#556b2f',
        'darkorange'              => '#ff8c00',
        'darkorchid'              => '#9932cc',
        'darkred'                 => '#8b0000',
        'darksalmon'              => '#e9967a',
        'darkseagreen'            => '#8fbc8f',
        'darkslateblue'           => '#483d8b',
        'dark slateyellow'        => '#2f4f4f',
        'darkturquoise'           => '#00ced1',
        'darkviolet'              => '#9400d3',
        'deeppink'                => '#ff1493',
        'deepskyblue'             => '#00bfff',
        'dimgrey'                 => '#696969',
        'dodgerblue'              => '#1e90ff',
        'firebrick'               => '#b22222',
        'floralwhite'             => '#fffaf0',
        'forestgreen'             => '#228b22',
        'fuchsia'                 => '#ff00ff',
        'gainsboro'               => '#dcdcdc',
        'ghostwhite'              => '#f8f8ff',
        'gold'                    => '#ffd700',
        'goldenrod'               => '#daa520',
        'gray'                    => '#808080',
        'green'                   => '#008000',
        'greenyellow'             => '#adff2f',
        'grey'                    => '#808080',
        'honeydew'                => '#f0fff0',
        'hotpink'                 => '#ff69b4',
        'indianred'               => '#cd5c5c',
        'indigo'                  => '#4b0082',
        'ivory'                   => '#fffff0',
        'khaki'                   => '#f0e68c',
        'lavender'                => '#e6e6fa',
        'lavenderblush'           => '#fff0f5',
        'lawngreen'               => '#7cfc00',
        'lemonchiffon'            => '#fffacd',
        'lightblue'               => '#add8e6',
        'lightcoral'              => '#f08080',
        'lightcyan'               => '#e0ffff',
        'light cyan'              => '#e0ffff',
        'cyan light'              => '#e0ffff',
        'magenta light'           => '#FFCCFF',
        'lightmagenta'            => '#FFCCFF',
        'light magenta'           => '#FFCCFF',
        'light goldenrod yellow'  => '#fafad2',
        'lightgray'               => '#d3d3d3',
        'lightgreen'              => '#90ee90',
        'lightgrey'               => '#d3d3d3',
        'lightpink'               => '#ffb6c1',
        'lightsalmon'             => '#ffa07a',
        'lightseagreen'           => '#20b2aa',
        'lightskyblue'            => '#87cefa',
        'lightslategrey'          => '#778899',
        'lightsteelblue'          => '#b0c4de',
        'lightyellow'             => '#ffffe0',
        'yellow light'            => '#ffffe0',
        'lime'                    => '#00ff00',
        'limegreen'               => '#32cd32',
        'linen'                   => '#faf0e6',
        'magenta'                 => '#ff00ff',
        'maroon'                  => '#800000',
        'medium aquamarine'       => '#66cdaa',
        'medium blue'             => '#0000cd',
        'medium orchid'           => '#ba55d3',
        'medium purple'           => '#9370db',
        'medium seagreen'         => '#3cb371',
        'medium slateblue'        => '#7b68ee',
        'medium springgreen'      => '#00fa9a',
        'medium turquoise'        => '#48d1cc',
        'medium violetred'        => '#c71585',
        'midnightblue'            => '#191970',
        'mintcream'               => '#f5fffa',
        'mistyrose'               => '#ffe4e1',
        'moccasin'                => '#ffe4b5',
        'navajowhite'             => '#ffdead',
        'navy'                    => '#000080',
        'oldlace'                 => '#fdf5e6',
        'olive'                   => '#808000',
        'olivedrab'               => '#6b8e23',
        'orange'                  => '#ffa500',
        'orangered'               => '#ff4500',
        'orchid'                  => '#da70d6',
        'palegoldenrod'           => '#eee8aa',
        'palegreen'               => '#98fb98',
        'paleturquoise'           => '#afeeee',
        'palevioletred'           => '#db7093',
        'papayawhip'              => '#ffefd5',
        'peachpuff'               => '#ffdab9',
        'peru'                    => '#cd853f',
        'pink'                    => '#ffc0cb',
        'plum'                    => '#dda0dd',
        'powderblue'              => '#b0e0e6',
        'purple'                  => '#800080',
        'lila'                    => '#800080',
        'red'                     => '#ff0000',
        'rosybrown'               => '#bc8f8f',
        'royalblue'               => '#4169e1',
        'saddlebrown'             => '#8b4513',
        'salmon'                  => '#fa8072',
        'sandybrown'              => '#f4a460',
        'seagreen'                => '#2e8b57',
        'seashell'                => '#fff5ee',
        'sienna'                  => '#a0522d',
        'silver'                  => '#c0c0c0',
        'skyblue'                 => '#87ceeb',
        'slateblue'               => '#6a5acd',
        'slategrey'               => '#708090',
        'snow'                    => '#fffafa',
        'springgreen'             => '#00ff7f',
        'steelblue'               => '#4682b4',
        'tan'                     => '#d2b48c',
        'teal'                    => '#008080',
        'thistle'                 => '#d8bfd8',
        'tomato'                  => '#ff6347',
        'turquoise'               => '#40e0d0',
        'violet'                  => '#ee82ee',
        'wheat'                   => '#f5deb3',
        'white'                   => '#ffffff',
        'whitesmoke'              => '#f5f5f5',
        'yellow'                  => '#ffff00',
        'yellowgreen'             => '#9acd32',
        '3-farbig'                => ['#00ffff', '#ff00ff', '#ffff00'],
        'rgb'                     => ['#00ffff', '#ff00ff', '#ffff00'],
        'farbig'                  => ['#00ffff', '#ff00ff', '#ffff00'],
        'cmy'                     => ['#e0ffff', '#ff00ff', '#ffff00'],
        'cmyk'                    => ['#e0ffff', '#ff00ff', '#000000'],
        '4-farbig'                => ['#00ffff', '#ff00ff', '#ffff00', '#000000'],
        '5-farbig'                => ['#00ffff', '#ff00ff', '#ffff00', '#e0ffff', '#FFCCFF'],
        'color'                   => ['#00ffff', '#ff00ff', '#ffff00'],
        'weiss'                   => '#ffffff',
        'chamois'                 => '#efe0b9',
        'lichtgrau'               => '#c4caca',
        'türkisblau'              => '#21888f',
        'bordeaux'                => '#8D273C',
        'xenon'                   => '#14397f'
    ];


    /**
     * Cache for the user-defined color overrides from the plugin configuration.
     */
    private ?array $customOverridesMap = null;

    public function __construct(
        private readonly SystemConfigService $systemConfigService
    ) {
    }

    /**
     * A reusable, public method to resolve a color name to its CSS value.
     * This is ideal for external use, like in a connector plugin, to determine
     * if a `colorHexCode` should be set on a property during import.
     *
     * TODO: use this in the connector plugin
     *
     * @param string $colorName The color name to resolve (e.g., "Rot", "cmyk").
     * @return string|array|null The resolved CSS color string, an array of strings for multi-color values (e.g. "cymk"), or null if no mapping is found.
     */
    public function resolveColorValue(string $colorName): string|array|null
    {
        $lowerColorName = strtolower(trim($colorName));
        if (empty($lowerColorName)) {
            return null;
        }

        $customOverrides = $this->_getCustomOverridesMap();

        // Priority 1: Check custom overrides from plugin config
        if (isset($customOverrides[$lowerColorName])) {
            return $customOverrides[$lowerColorName];
        }

        // Priority 2: Check built-in default map
        if (isset(self::DEFAULT_MAP_COLOR_NAME_TO_CSS_COLOR[$lowerColorName])) {
            return self::DEFAULT_MAP_COLOR_NAME_TO_CSS_COLOR[$lowerColorName];
        }

        return null;
    }

    /**
     * Get and cache the custom color overrides from the plugin configuration.
     */
    private function _getCustomOverridesMap(): array
    {
        if ($this->customOverridesMap === null) {
            $this->customOverridesMap = [];
            $colorOverwriteConfig = $this->systemConfigService->get('TopdataColors.config.colorOverwrite');

            if (!empty($colorOverwriteConfig)) {
                $lines = explode("\n", $colorOverwriteConfig);
                foreach ($lines as $line) {
                    $parts = explode('|', trim($line));
                    if (count($parts) === 2 && !empty($parts[0])) {
                        // The value can be a single color code, or multiple codes separated by '/'
                        $colorValues = explode('/', trim($parts[1]));
                        $this->customOverridesMap[strtolower($parts[0])] = (count($colorValues) > 1) ? $colorValues : $colorValues[0];
                    }
                }
            }
        }

        return $this->customOverridesMap;
    }

    /**
     * Get color values and existing color codes for a product to display in the storefront.
     */
    public function getColorValuesAndExistingColorCodes(SalesChannelProductEntity $product): array
    {
        $colorPropertyGroups = $this->systemConfigService->get('TopdataColors.config.colorPropertyGroups');
        if (empty($colorPropertyGroups) || !$product->getProperties()) {
            return [];
        }

        $customOverridesMap = $this->_getCustomOverridesMap();
        $finalColorGroups = [];

        // Collect raw property data first
        $propertiesToProcess = [];
        foreach ($colorPropertyGroups as $groupId) {
            foreach ($product->getProperties() as $property) {
                if ($property->getGroupId() === $groupId) {
                    $propertiesToProcess[] = [
                        'hexCode'     => $property->getColorHexCode(),
                        'name'        => $property->getName(), // Use original name for logic/parsing
                        'displayName' => $property->getTranslated()['name'] ?? $property->getName()
                    ];
                }
            }
        }

        if (empty($propertiesToProcess)) {
            return [];
        }

        // Now, process the collected properties
        foreach ($propertiesToProcess as $property) {
            $displayName = $property['displayName'];
            $subGroups = explode(',', $property['name']);

            foreach ($subGroups as $subGroup) {
                $subGroup = trim($subGroup);
                if (empty($subGroup)) {
                    continue;
                }

                $quantity = 1;
                $colorName = $subGroup;

                if (preg_match('/^(\d+)x\s+(.+)$/i', $subGroup, $matches)) {
                    $quantity = (int)$matches[1];
                    $colorName = trim($matches[2]);
                }

                $resolvedColors = [];
                $colorParts = (strpos($colorName, '+') !== false) ? explode('+', $colorName) : [$colorName];
                $isSingleColorProperty = (count($colorParts) === 1);

                foreach ($colorParts as $singleColorPart) {
                    $singleColorPart = trim($singleColorPart);
                    if (empty($singleColorPart)) continue;

                    $lowerSingleColor = strtolower($singleColorPart);
                    $finalColorValue = null;

                    // --- Start Resolution Logic with specified precedence for storefront display ---
                    // Step 1: Baseline from default map
                    if (isset(self::DEFAULT_MAP_COLOR_NAME_TO_CSS_COLOR[$lowerSingleColor])) {
                        $finalColorValue = self::DEFAULT_MAP_COLOR_NAME_TO_CSS_COLOR[$lowerSingleColor];
                    }

                    // Step 2: Override with property's hex code if it exists and this is a single-color property
                    if ($isSingleColorProperty && !empty($property['hexCode'])) {
                        $finalColorValue = $property['hexCode'];
                    }

                    // Step 3: Highest priority override from plugin config
                    if (isset($customOverridesMap[$lowerSingleColor])) {
                        $finalColorValue = $customOverridesMap[$lowerSingleColor];
                    }

                    if ($finalColorValue) {
                        if (is_array($finalColorValue)) {
                            $resolvedColors = array_merge($resolvedColors, $finalColorValue);
                        } else {
                            $resolvedColors[] = $finalColorValue;
                        }
                    } else {
                        $resolvedColors[] = $singleColorPart; // Fallback
                    }
                }

                if (!empty($resolvedColors)) {
                    $finalColorGroups[] = [
                        'quantity'    => $quantity,
                        'colors'      => $resolvedColors,
                        'displayName' => $displayName
                    ];
                }
            }
        }
        return $finalColorGroups;
    }
}
