<?php declare(strict_types=1);

namespace Topdata\TopdataCategorySearchSW6\Subscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\Suggest\SuggestPageLoadedEvent;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\OrFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Content\Category\CategoryDefinition;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\RangeFilter;
use Shopware\Core\System\SystemConfig\SystemConfigService;

/**
 * Subscriber class to add category suggestions to the storefront suggest page.
 */
class Subscriber implements EventSubscriberInterface
{
    private ?array $settings = null;
    protected SalesChannelRepository $categoryRepository;
    protected SystemConfigService $systemConfigService;

    public function __construct(
        SalesChannelRepository $categoryRepository,
        SystemConfigService    $systemConfigService
    )
    {
        $this->categoryRepository = $categoryRepository;
        $this->systemConfigService = $systemConfigService;
    }

    /**
     * Returns the events to which this class subscribes.
     *
     * @return array<string, string>
     */
    public static function getSubscribedEvents()
    {
        return [
            SuggestPageLoadedEvent::class => 'addCategoriesSuggest',
        ];
    }

    /**
     * Adds category suggestions to the suggest page.
     *
     * @param SuggestPageLoadedEvent $event
     */
    public function addCategoriesSuggest(SuggestPageLoadedEvent $event): void
    {
        $term = mb_strtolower(trim($event->getRequest()->query->get('search')));
        if (mb_strlen($term) > 2) {
            // ---- Set the plugin settings for the current sales channel
            $this->setSettings($event->getSalesChannelContext()->getSalesChannel());
            // ---- Fetch category suggestions based on the search term
            $categories = $this->suggestCategories($term, $event->getSalesChannelContext());
            if ($categories->getTotal() == 0) {
                return;
            }
            // ---- Add the category suggestions to the suggest page as an extension
            $event->getPage()->addExtension(
                'categories_suggest',
                $categories
//                new ArrayEntity([
//                        $categories->getEntities()
//                    ])
            );
        }
    }


    /**
     * Retrieves category suggestions based on the search term and sales channel context.
     *
     * @param string $term
     * @param SalesChannelContext $salesChannelContext
     * @return EntitySearchResult
     */
    private function suggestCategories(string $term, SalesChannelContext $salesChannelContext): EntitySearchResult
    {
        $currentSalesChannelCategoryId = $salesChannelContext->getSalesChannel()->getNavigationCategoryId();

        $filters = [];
        // ---- Split the search term into subterms and filter out empty strings
        $subterms = \array_filter(\explode(' ', $term), static function (string $subterm) {
            return \trim($subterm) !== '';
        });
        // ---- Iterate over the subterms and create filters based on the plugin settings
        foreach ($subterms as $subterm) {
            $subfilters = [];
            if ($this->getSetting('searchName')) {
                $subfilters[] = new ContainsFilter('name', $subterm);
            }
            if ($this->getSetting('searchTags')) {
                $subfilters[] = new ContainsFilter('tags.name', $subterm);
            }
            if ($this->getSetting('searchDescription')) {
                $subfilters[] = new ContainsFilter('description', $subterm);
            }
            if ($this->getSetting('searchMetaTitle')) {
                $subfilters[] = new ContainsFilter('metaTitle', $subterm);
            }
            if ($this->getSetting('searchMetaDescription')) {
                $subfilters[] = new ContainsFilter('metaDescription', $subterm);
            }
            if ($this->getSetting('searchKeywords')) {
                $subfilters[] = new ContainsFilter('keywords', $subterm);
            }

            if (count($subfilters) == 0) {
                $subfilters[] = new ContainsFilter('name', $subterm);
            }

            $filters = \array_merge($filters, $subfilters);
        }

        // ---- Create the criteria for the category search
        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('active', true))
            ->addFilter(new EqualsFilter('type', CategoryDefinition::TYPE_PAGE))
            ->addFilter(new RangeFilter('level', [RangeFilter::GT => 1]))
            ->addFilter(new ContainsFilter('path', $currentSalesChannelCategoryId))
            ->addFilter(new OrFilter($filters))
            ->setLimit((int)$this->getSetting('limit'));

        // ---- Add sorting to the criteria based on the plugin settings
        if ($this->getSetting('sort') == 'level') {
            $criteria = $criteria
                ->addSorting(new FieldSorting('level', FieldSorting::ASCENDING))
                ->addSorting(new FieldSorting('name', FieldSorting::ASCENDING));
        } else {
            $criteria = $criteria
                ->addSorting(new FieldSorting('name', FieldSorting::ASCENDING));
        }

        // ---- Search for categories using the criteria
        $categories = $this
            ->categoryRepository
            ->search($criteria, $salesChannelContext);

        return $categories;
    }

    /**
     * Sets the plugin settings for the current sales channel.
     *
     * @param $salesChannel
     */
    private function setSettings($salesChannel)
    {
        $this->settings = $this->systemConfigService->get(
            'TopdataCategorySearchSW6.config',
            $salesChannel->getId()
        );
    }

    /**
     * Retrieves a specific plugin setting.
     *
     * @param string $key
     * @return mixed
     */
    private function getSetting(string $key)
    {
        if (!isset($this->settings)) {
            $this->settings = $this->systemConfigService->get('TopdataCategorySearchSW6.config');
        }

        return isset($this->settings[$key]) ? $this->settings[$key] : false;
    }
}