import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';

export default class TdxFilterStatePlugin extends Plugin {
    static options = {
        initialFilterState: 'collapsed' // Default fallback value
    };

    init() {
        console.log('[TdxFilterStatePlugin] Initializing...');

        // Find our hidden config element in the DOM
        const configElement = DomAccess.querySelector(document, '#tdx-filter-config', false);
        console.log('[TdxFilterStatePlugin] Found config element:', configElement);

        // If the element exists, read the data attribute and update the plugin options
        if (configElement && configElement.dataset.initialState) {
            const stateFromConfig = configElement.dataset.initialState;
            console.log(`[TdxFilterStatePlugin] Read initial state from config: "${stateFromConfig}"`);
            this.options.initialFilterState = stateFromConfig;
        } else {
            console.warn('[TdxFilterStatePlugin] Could not find config element #tdx-filter-config or data-initial-state attribute. Using default.');
        }

        console.log(`[TdxFilterStatePlugin] Final state to apply: "${this.options.initialFilterState}"`);

        // Execute the core logic
        this._applyInitialFilterState();
    }

    _applyInitialFilterState() {
        const state = this.options.initialFilterState;

        if (state === 'expanded') {
            console.log('[TdxFilterStatePlugin] State is "expanded", attempting to expand all filters.');
            this._expandAllFilterAccordions();
        } else {
            console.log('[TdxFilterStatePlugin] State is "collapsed", no action needed.');
        }
    }

    _expandAllFilterAccordions() {
        // Find all filter panel toggle buttons that are not already expanded
        const selector = '.filter-panel-item-toggle[data-bs-toggle="collapse"][aria-expanded="false"]';

        const toggleButtons = document.querySelectorAll(selector);

        console.log(`[TdxFilterStatePlugin] Found ${toggleButtons.length} filter toggle buttons with selector: "${selector}"`, toggleButtons);

        if (toggleButtons.length === 0) {
            console.warn('[TdxFilterStatePlugin] Warning: No collapsed filter accordions were found to expand. Check if the theme uses a different class or structure.');
        }

        toggleButtons.forEach((button, index) => {
            const targetSelector = button.getAttribute('data-bs-target');
            if (!targetSelector) {
                console.error(`[TdxFilterStatePlugin] Button ${index} is missing a 'data-bs-target' attribute.`, button);
                return;
            }

            const targetElement = document.querySelector(targetSelector);
            if (targetElement && !targetElement.classList.contains('show')) {
                console.log(`[TdxFilterStatePlugin] Expanding target: "${targetSelector}"`, { button, targetElement });

                // Use Bootstrap's own methods if available, otherwise fallback to manual class manipulation
                const bsCollapse = window.bootstrap ? window.bootstrap.Collapse.getInstance(targetElement) : null;
                if (bsCollapse) {
                    bsCollapse.show();
                } else {
                    console.warn(`[TdxFilterStatePlugin] Bootstrap Collapse instance not found for "${targetSelector}". Falling back to manual class manipulation.`);
                    targetElement.classList.add('show');
                    button.classList.remove('collapsed');
                    button.setAttribute('aria-expanded', 'true');
                }
            } else {
                console.log(`[TdxFilterStatePlugin] Target "${targetSelector}" already expanded or not found.`);
            }
        });
    }
}