<?php

namespace Topdata\TopdataCategoryFilterSW6\Elasticsearch\Product;


# use OpenSearchDSL\Query\Compound\BoolQuery; # 6.5
use ONGR\ElasticsearchDSL\Query\Compound\BoolQuery; # 6.4
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityDefinition;
use Shopware\Elasticsearch\Framework\AbstractElasticsearchDefinition;
use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;

/**
 * This class extends the AbstractElasticsearchDefinition class and modifies the Elasticsearch mapping
 * for products by adding a nested field 'streamCategories' with an 'id' property.
 * It also provides methods to fetch dynamic categories and extend entities and criteria.
 */
class ProductEsDecorator extends AbstractElasticsearchDefinition
{
    private AbstractElasticsearchDefinition $productDefinition;
    private Connection $connection;

    public function __construct(AbstractElasticsearchDefinition $productDefinition, Connection $connection)
    {
        $this->productDefinition = $productDefinition;
        $this->connection = $connection;
    }

    public function getEntityDefinition(): EntityDefinition
    {
        return $this->productDefinition->getEntityDefinition();
    }

    /**
     * Returns the mapping for the Elasticsearch index.
     *
     * This method extends the mapping from the parent product definition
     * by adding a nested field 'streamCategories' with an 'id' property.
     *
     * @param Context $context The context.
     * @return array The mapping for the Elasticsearch index.
     */
    public function getMapping(Context $context): array
    {
        $mapping = $this->productDefinition->getMapping($context);

        // Adding a nested field with id
        $mapping['properties']['streamCategories'] = [
            'type'       => 'nested',
            'properties' => [
                'id' => [
                    'type'       => 'keyword',
                    'normalizer' => 'sw_lowercase_normalizer',
                ],
            ],
        ];

        return $mapping;
    }

    public function buildTermQuery(Context $context, Criteria $criteria): BoolQuery
    {
        return $this->productDefinition->buildTermQuery($context, $criteria);
    }

    /**
     * Fetches documents for the given product IDs and adds dynamic categories to each document.
     *
     * @param array $productIds The array of product IDs to fetch documents for.
     * @param Context $context The context in which the fetch operation is being performed.
     * @return array The array of fetched documents with added dynamic categories.
     */
    public function fetch(array $productIds, Context $context): array
    {
        $documents = $this->productDefinition->fetch($productIds, $context);

        $dynamicCategories = $this->fetchDynamicCategories($productIds);

        foreach ($documents as &$document) {
            if (isset($dynamicCategories[$document['id']])) {
                $document['streamCategories'] = array_map(function (string $id) {
                    return ['id' => $id];
                }, array_filter(explode('|', $dynamicCategories[$document['id']] ?? '')));
            }
        }

        return $documents;
    }

    /**
     * Fetches dynamic categories based on provided Product IDs.
     *
     * @param string[] $productIds The array of Product IDs to fetch dynamic categories for.
     * @return array Fetched dynamic categories as an array.
     */
    private function fetchDynamicCategories(array $productIds): array
    {
        $query = <<<SQL
            SELECT LOWER(HEX(s.product_id)) as pid, GROUP_CONCAT(LOWER(HEX(c.id)) SEPARATOR "|") as cids
            FROM category c, product_stream_mapping s
            WHERE
                c.product_stream_id = s.product_stream_id
                AND s.product_id IN(:productIds)
            GROUP BY s.product_id
        SQL;


        return $this->connection->fetchAllKeyValue(
            $query,
            [
                'productIds' => $productIds,
            ],
            [
                'ids' => Connection::PARAM_STR_ARRAY
            ]
        );
    }

    public function extendEntities(EntityCollection $collection): EntityCollection
    {
        return $this->productDefinition->extendEntities($collection);
    }

    public function extendCriteria(Criteria $criteria): void
    {
        $this->productDefinition->extendCriteria($criteria);
    }
}