<?php

declare(strict_types=1);

namespace Sisi\Search\Storefront\Page;

use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Content\Product\SalesChannel\Search\AbstractProductSearchRoute;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\Routing\Exception\MissingRequestParameterException;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoaderInterface;
use Shopware\Storefront\Page\Search\SearchPage;
use Shopware\Storefront\Page\Suggest\SuggestPage;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * This class is responsible for loading the search page and suggest page.
 * It uses the generic page loader to load the base page and then adds the search specific data.
 */
class SearchPageLoader
{
    /**
     * @var GenericPageLoaderInterface
     */
    private $genericLoader;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var AbstractProductSearchRoute
     */
    private $productSearchRoute;

    /**
     * @param GenericPageLoaderInterface $genericLoader
     * @param AbstractProductSearchRoute $productSearchRoute
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        GenericPageLoaderInterface $genericLoader,
        AbstractProductSearchRoute $productSearchRoute,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->genericLoader = $genericLoader;
        $this->productSearchRoute = $productSearchRoute;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * Loads the search page with the provided request and sales channel context.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @return SearchPage The loaded search page.
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     * @throws MissingRequestParameterException
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext)
    {
        // ---- Load the generic page
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $page = SearchPage::createFrom($page);

        // ---- Set the search term from the request
        $page->setSearchTerm(
            (string)$request->query->get('search')
        );
        return $page;
    }

    /**
     * Loads the suggest page with the provided request and sales channel context.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @return SuggestPage The loaded suggest page.
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     * @throws MissingRequestParameterException
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function loadSuggest(Request $request, SalesChannelContext $salesChannelContext)
    {
        // ---- Load the generic page
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $page = SuggestPage::createFrom($page);

        // ---- Set the search term from the request
        $page->setSearchTerm(
            (string)$request->query->get('search')
        );
        return $page;
    }


    /**
     * Loads a generic page.
     *
     * @param Request $request The request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @return \Shopware\Storefront\Page\Page
     */
    public function getPage(Request $request, SalesChannelContext $salesChannelContext)
    {
        return $this->genericLoader->load($request, $salesChannelContext);
    }
}