<?php

namespace Sisi\Search\ServicesInterfaces;

use Elasticsearch\Client;
use Shopware\Core\Content\Category\CategoryEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;

/**
 * Interface InterfaceSearchCategorieService
 *
 * This interface defines the methods for searching categories within the system,
 * including searching in the default index and in a custom index. It also provides
 * methods for inserting category data into Elasticsearch and creating search criteria
 * and mappings.
 */
interface InterfaceSearchCategorieService
{
    /**
     * Searches for categories based on the provided parameters.
     *
     * @param array $config The general configuration array.
     * @param array $params The search parameters.
     * @param array $configkategorie The category-specific configuration.
     * @param Client $client The Elasticsearch client.
     * @param string $term The search term.
     * @return array|void Returns an array of search results or void if an error occurs.
     */
    public function searchCategorie($config, $params, $configkategorie, $client, $term);

    /**
     * Searches for categories using a custom Elasticsearch index.
     *
     * @param array $config The general configuration array.
     * @param array $params The search parameters.
     * @param array $configkategorie The category-specific configuration.
     * @param Client $client The Elasticsearch client.
     * @param string $term The search term.
     * @return array|void Returns an array of search results or void if an error occurs.
     */
    public function searchCategorieWithOwnIndex($config, $params, $configkategorie, $client, $term);

    /**
     * Inserts category data into Elasticsearch.
     *
     * @param Client $client The Elasticsearch client.
     * @param string $indexname The name of the Elasticsearch index.
     * @param CategoryEntity $category The category entity to insert.
     * @param array $fieldConfig The field configuration.
     * @param array $config The general configuration.
     * @param array $parameter Additional parameters.
     * @return array Returns an array containing the result of the insertion.
     */
    public function insertValue($client, $indexname, $category, $fieldConfig, $config, $parameter);

    /**
     * Creates a new Criteria object for category searches.
     *
     * @return Criteria
     */
    public function createCriteria(): Criteria;

    /**
     * Creates the mapping for the category index in Elasticsearch.
     *
     * @param array $fieldConfig The field configuration.
     * @return array Returns the mapping configuration as an array.
     */
    public function createCategoryMapping(array $fieldConfig): array;

    /**
     * Creates the settings for the category index in Elasticsearch.
     *
     * @SuppressWarnings(PHPMD)
     * @param array $fieldConfigs The field configurations.
     * @param array $config The general configuration.
     * @return void
     */
    public function createCategorySettings(array $fieldConfigs, array $config);
}