<?php

namespace Sisi\Search\Service;

use Shopware\Core\Content\Media\Pathname\UrlGeneratorInterface;
use Shopware\Core\Content\Product\ProductCollection;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\PropertyGroupCollection;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Pricing\PriceCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Sisi\Search\Service\ContextService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This class provides services related to product variants,
 * including setting up database queries, fixing mappings,
 * inserting data into Elasticsearch, and modifying search queries.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class VariantenService
{

    private $stock = 0;

    /**
     * Sets up the database query criteria to include associated data for product variants.
     *
     * This method modifies the given Criteria object to include associations necessary
     * for fetching product variant data.  It checks a configuration setting to determine
     * whether to include the 'cheapestPrice' association.
     *
     * @param array $config Configuration array.
     * @param Criteria $criteria The criteria object to modify.
     */
    public function setDBQueryWithvariants(array $config, Criteria &$criteria): void
    {
        if ($this->conditionFunction($config)) {
            $criteria->addAssociation('children');
            $criteria->addAssociation('children.properties.translations');
            $criteria->addAssociation('children.properties.group');
            $criteria->addAssociation('children.properties.group.translations');
            $criteria->addAssociation('children.cover');
            $criteria->addAssociation('children.translations');
            if (array_key_exists('cheapestPrice', $config)) {
                if ($config['cheapestPrice'] === 'yes') {
                    $criteria->addAssociation('children.cheapestPrice');
                    $criteria->addAssociation('cheapestPrice');
                }
            }
        }
    }

    /**
     * Fixes the Elasticsearch mapping for product variants.
     *
     * This method modifies the Elasticsearch mapping to include nested properties
     * for product variants. It iterates through the provided variant fields and
     * creates a nested mapping for each field.
     *
     * @param array $config Configuration array.
     * @param array $mapping The Elasticsearch mapping to modify.
     * @param array $variantsFields Array of variant fields.
     */
    public function fixMappingForvariants(array $config, &$mapping, $variantsFields)
    {
        if ($this->conditionFunction($config)) {
            $properties = [];
            foreach ($variantsFields as $key => $pro) {
                $name = $pro->getPrefix() . "product_" . $pro->getName();
                $type = $pro->getFieldtype();
                $properties[$name] = [
                    "type" => "nested",
                ];
                if ($type === 'text') {
                    $properties[$name] = [
                        "type" => $type,
                        "analyzer" => "analyzer_" . $name,
                    ];
                }
            }
            $mapping['properties']['children'] = [
                "type" => "nested",
                'properties' => $properties,
            ];
            $mapping['properties']['children']['properties']['properties'] = [
                "type" => "nested",
            ];
        }
    }

    /**
     * Fixes the Elasticsearch insert data for product variants.
     *
     * This method prepares and inserts data for product variants into Elasticsearch.
     * It retrieves child products, merges their data, and updates the main product's
     * stock information.
     *
     * @param array $fields The fields array to modify.
     * @param SalesChannelProductEntity $entity The product entity.
     * @param array $config Configuration array.
     * @param array $fieldsconfig Field configuration array.
     * @param InsertService $self The InsertService instance.
     * @param array $lanuagesArray Array of language IDs.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param ContainerInterface $container The container interface.
     * @param array $merkerIdsFordynamicProducts Array to store dynamic product IDs.
     * @param UrlGeneratorInterface $urlGenerator The URL generator.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function fixEsInsertForvariants(
        array &$fields,
        SalesChannelProductEntity &$entity,
        array $config,
        array $fieldsconfig,
        InsertService $self,
        array $lanuagesArray,
        SalesChannelContext $saleschannelContext,
        ContainerInterface $container,
        array &$merkerIdsFordynamicProducts,
        UrlGeneratorInterface $urlGenerator
    ): void {
        if ($this->conditionFunction($config)) {
            $strLoop = true;
            $count = 0;
            $setpSize = 100;
            $fields['children'] = [];

            // ---- Get step size from config
            if (array_key_exists('childstepsize', $config)) {
                $setpSize = $config['childstepsize'];
            }
            $index = 0;
            $this->stock = 0;

            // ---- Loop through children in steps
            while ($strLoop) {
                $childrenObject = $this->getChildren(
                    $container,
                    $entity->getId(),
                    $saleschannelContext,
                    $setpSize,
                    $count
                );
                $total = (int) $childrenObject->getTotal();
                $count = $count + $total;

                // ---- Merge children variants
                $this->mergeCildrenVariants(
                    $fields,
                    $entity,
                    $config,
                    $fieldsconfig,
                    $self,
                    $lanuagesArray,
                    $merkerIdsFordynamicProducts,
                    $childrenObject,
                    $urlGenerator,
                    $index
                );
                if ($total == 0) {
                    $strLoop = false;
                }
            }

            // ---- Update product stock
            if (array_key_exists('product_stock', $fields)) {
                $fields['product_stock'] = $this->stock;
            }
        }
    }

    /**
     * Merges children variants into the fields array for Elasticsearch insertion.
     *
     * This method iterates through the children entities, extracts relevant data,
     * and merges it into the main product's fields array for Elasticsearch insertion.
     * It handles configurator settings, translations, stock information, and property
     * assignments.
     *
     * @param array $fields The fields array to modify.
     * @param SalesChannelProductEntity $entity The product entity.
     * @param array $config Configuration array.
     * @param array $fieldsconfig Field configuration array.
     * @param InsertService $self The InsertService instance.
     * @param array $lanuagesArray Array of language IDs.
     * @param array $merkerIdsFordynamicProducts Array to store dynamic product IDs.
     * @param EntitySearchResult<SalesChannelProductEntity> $childern The children entities.
     * @param UrlGeneratorInterface $urlGenerator The URL generator.
     * @param int $index The index.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    private function mergeCildrenVariants(
        array &$fields,
        SalesChannelProductEntity &$entity,
        array $config,
        array $fieldsconfig,
        InsertService $self,
        array $lanuagesArray,
        array &$merkerIdsFordynamicProducts,
        EntitySearchResult $childern,
        UrlGeneratorInterface $urlGenerator,
        int &$index
    ): void {
        if (count($fields) > 0) {
            /** @var array<array<string>> $configuratorSettings */
            $configuratorSettings = $entity->getConfiguratorGroupConfig();
            $heandlerkeyword = new SearchkeyService();
            $heandlerFixurl = new ChannelDataService();
            $mainId = $entity->getMainVariantId();
            if ($mainId !== null && !empty($mainId) && $config['addVariants'] === 'individual') {
                $merkerIdsFordynamicProducts[] = $mainId;
            }
            $pricecolltion = new PriceCollection();
            $sorted = new PropertyGroupCollection();
            $optionen = [];
            $mainPropeties = $entity->getSortedProperties();
            foreach ($mainPropeties->getElements() as $prop) {
                $sorted->add($prop);
                foreach ($prop->getOptions()->getElements() as $key => $op) {
                    $optionen[$key] = $op;
                }
            }
            foreach ($entity->getPrice()->getElements() as $price) {
                $pricecolltion->add($price);
            }
            $strListing = false;
            if ($configuratorSettings !== null && $config['addVariants'] === 'individual') {
                foreach ($configuratorSettings as $configuratorSettingItem) {
                    if ($configuratorSettingItem["expressionForListings"]) {
                        echo "Item is listed \n";
                        $strListing = true;
                    }
                }
            }
            if ($config['addVariants'] === 'yes') {
                $translations = $entity->getTranslations();
                foreach ($translations as $translation) {
                    $customFields = $translation->getCustomFields();
                    if ($customFields !== null) {
                        foreach ($customFields as $name => $value) {
                            if ($name === 'sisi_list' && $value > 0) {
                                $strListing = true;
                            }
                        }
                    }
                }
                if (array_key_exists('checkvariantsName', $config)) {
                    if ($config['checkvariantsName'] === 'yes') {
                        $merkerheader = [];
                        $translations = $entity->getTranslated();
                        $translations[] = $translations['name'];
                        foreach ($childern as $child) {
                            $translations = $child->getTranslated();
                            if (!in_array($translations['name'], $merkerheader)) {
                                $merkerheader[] = $translations['name'];
                            }
                        }
                        if (count($merkerheader) > 1) {
                            $strListing = true;
                        }
                    }
                }
            }

            // ---- Loop through languages
            foreach ($lanuagesArray as $languageId) {
                // ---- Loop through children
                foreach ($childern as $key => $child) {
                    // check the variants will be as listing
                    if ($strListing) {
                        $merkerIdsFordynamicProducts[] = $child->getId();
                    }
                    $strStock = true;
                    if (array_key_exists('product_stock', $fields)) {
                        $strStock = false;
                        $stock = $child->getAvailableStock();
                        $this->stock = $this->stock + $stock;
                        if ($stock > 0) {
                            $strStock = true;
                        }
                    }
                    if (!$strListing && $strStock) {
                        $fieldsConfigValues = $this->varaintenValue(
                            $fieldsconfig,
                            $config,
                            $child,
                            $languageId,
                            $self
                        );
                        foreach ($fieldsConfigValues as $fieldsItem) {
                            if (array_key_exists('indexName', $fieldsItem)) {
                                $indexName = $fieldsItem['indexName'];
                                $fields['children'][$index][$indexName] = $fieldsItem['value'];
                            }
                        }
                        $fields['children'][$index]['product_id'] = $child->getId();
                        $fields['children'][$index]['price'] = $child->getPrice();
                        $heandlerFixurl->fixMediaUrl($child, $urlGenerator, $config);
                        $cover = $child->getCover();
                        if ($cover === null) {
                            $cover = $entity->getCover();
                        }
                        $fields['children'][$index]['cover'] = $cover;
                        $sortvalue = $child->getSortedProperties();
                        $heandlerkeyword->insertCustomSearchKey($config, $child, $fields['children'][$index]);
                        if ($sortvalue !== null) {
                            foreach ($sortvalue->getElements() as $prop) {
                                $sorted->add($prop);
                                foreach ($prop->getOptions()->getElements() as $key => $op) {
                                    $optionen[$key] = $op;
                                    $propName = $prop->getName();
                                    $optname = $op->getName();
                                    $fields['children'][$index]['properties'][] = [
                                        'option_id' => $op->getId(),
                                        'option_name' => $optname,
                                        'property_group' => $propName,
                                        'property_id' => $prop->getId(),
                                    ];
                                }
                            }
                        }
                        foreach ($child->getPrice()->getElements() as $price) {
                            $pricecolltion->add($price);
                        }

                        foreach ($sorted->getElements() as &$sortedItem) {
                            $option = $sortedItem->getOptions()->getElements();
                            foreach ($option as $optionitem) {
                                foreach ($optionen as $rember) {
                                    if ($optionitem->getGroupId() === $rember->getGroupId()) {
                                        $sortedItem->getOptions()->add($rember);
                                    }
                                }
                            }
                        }
                        if (array_key_exists('channel', $fields)) {
                            $fields['channel']->setPrice($pricecolltion);
                            $fields['channel']->setSortedProperties($sorted);
                        }
                        $index++;
                    }
                }
            }

            $strresetChildren = true;
            if (array_key_exists('strchildren', $config)) {
                if ($config['strchildren'] === 'yes') {
                    $strresetChildren = false;
                }
            }
            if ($strresetChildren) {
                /* set empty object because too big Data */
                $emptyChildren = new ProductCollection();
                $entity->setChildren($emptyChildren);
            }
        }
        // set the varinat cover image when the cover image is null from the main produkt
        if ($entity->getParentId() === null && $entity->getCover() === null) {
            $children = $entity->getChildren();
            if ($children !== null) {
                $firstChildren = $children->first();
                if ($firstChildren !== null) {
                    $cover = $firstChildren->getCover();
                    if ($cover !== null) {
                        $entity->setCover($cover);
                    }
                }
            }
        }
    }

    /**
     * Gets the children entities for a given product ID.
     *
     * This method retrieves a paginated list of child products associated with
     * the given product ID. It applies filters and associations to the criteria
     * to fetch the necessary data.
     *
     * @param ContainerInterface $container The container interface.
     * @param string $productId The product ID.
     * @param SalesChannelContext $context The sales channel context.
     * @param int $setpSize The step size.
     * @param int $offset The offset.
     * @return EntitySearchResult<SalesChannelProductEntity>
     */
    private function getChildren(ContainerInterface $container, string $productId, SalesChannelContext $context, int $setpSize, int $offset)
    {
        $criteria = new Criteria();
        $fieldsService = $container->get('sales_channel.product.repository');
        $criteria->addFilter(new EqualsFilter('parentId', $productId));
        $criteria->addAssociation('properties.translations');
        $criteria->addAssociation('properties.group');
        $criteria->addAssociation('properties.group.translations');
        $criteria->addAssociation('cover');
        $criteria->addAssociation('translations');
        $criteria->setLimit($setpSize);
        $criteria->setOffset($offset);
        return $fieldsService->search($criteria, $context);
    }

    /**
     * Gets the variant values for a given child entity.
     *
     * This method extracts and prepares variant-specific values from a child product
     * entity based on the provided field configurations and language.
     *
     * @param array $fieldsconfig The field configuration.
     * @param array $config The configuration array.
     * @param SalesChannelProductEntity $child The child entity.
     * @param string $languageId The language ID.
     * @param InsertService $self The InsertService instance.
     * @return array
     */
    private function varaintenValue($fieldsconfig, $config, $child, $languageId, $self)
    {
        $heandlerExtendSearch = new ExtSearchService();
        $haendlerTranslation = new TranslationService();
        $heandler = new ExtendInsertService();
        $index = 0;
        $return = [];
        foreach ($fieldsconfig as $mappingValue) {
            $name = 'get' . ucfirst($mappingValue->getName());
            $value = $child->$name();
            $value = $heandler->getValueFromDefaultLanguage($name, $child, $value);
            $translations = $child->getTranslations();
            if (!empty($translations)) {
                $translation = $haendlerTranslation->getTranslationfields($translations, strtolower($languageId), $config);
                if ($translation) {
                    if (method_exists($translation, $name)) {
                        $newvalue = $translation->$name();
                        if (!empty($newvalue)) {
                            $value = $newvalue;
                        }
                    }
                }
            }
            $value = $self->removeSpecialCharacters($value, $mappingValue);
            $value = $heandlerExtendSearch->stripUrl($value, $config);
            $indexName = $mappingValue->getPrefix() . 'product_' . $mappingValue->getName();
            $return[$index] = [
                'value' => $value,
                'indexName' => $indexName,
            ];
            $index++;
        }
        return $return;
    }

    /**
     * Gets the mapping product values.
     *
     * This method retrieves product mapping values based on the provided criteria.
     * It filters the fields based on the table name and whether they are only for variants.
     *
     * @param ContainerInterface $container The container interface.
     * @param Criteria $criteriaForFields The criteria for fields.
     * @return array
     */
    public function getMappingProductsValues(ContainerInterface $container, Criteria $criteriaForFields): array
    {
        $heandlerContext = new ContextService();
        $context = $heandlerContext->getContext();
        $fieldsService = $container->get('s_plugin_sisi_search_es_fields.repository');
        $criteriaForFields->addFilter(new EqualsFilter('tablename', 'product'));
        $criteriaForFields->addFilter(new EqualsFilter('onlymain', 'variante'));
        return $fieldsService->search($criteriaForFields, $context)->getEntities()->getElements();
    }

    /**
     * Changes the query for variant search.
     *
     * This method modifies the Elasticsearch query to include nested queries for
     * product variants. It iterates through the field configurations and creates
     * nested queries for each variant field.
     *
     * @param array $params The parameters array to modify.
     * @param string $term The search term.
     * @param array $fieldsconfig The field configuration.
     * @param array $config The configuration array.
     * @param bool $kindfilter Whether to apply kind filter.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    public function changeQueryForvariantssearch(
        array &$params,
        string $term,
        $fieldsconfig,
        array $config,
        bool $kindfilter
    ): void {
        $highlight = [];
        $shouldInner = [];
        $newTerm = $term;
        if ($this->conditionFunction($config)) {
            foreach ($fieldsconfig as $row) {
                if ($row->getOnlymain() === 'variante' && $row->getTablename() === 'product') {
                    $name = 'children.' . $row->getPrefix() . 'product_' . $row->getName();
                    $sonderValues = $row->getPhpfilter();
                    $sonderValuesArray = explode("\n", $sonderValues);
                    if (count($sonderValuesArray) > 0) {
                        foreach ($sonderValuesArray as $sonderValueitem) {
                            $newTerm = str_replace($sonderValueitem, "", $term);
                        }
                    }
                    $query['query'] = $newTerm;
                    $boost = $row->getBooster();
                    if (!empty($boost)) {
                        $query["boost"] = $boost;
                    } else {
                        unset($query["boost"]);
                    }
                    $fuzziness = $row->getFuzzy();
                    if (!empty($fuzziness)) {
                        $query["fuzziness"] = $fuzziness;
                    } else {
                        unset($query["fuzziness"]);
                    }
                    $operator = $row->getOperator();
                    if (!empty($operator)) {
                        $query["operator"] = $operator;
                    } else {
                        unset($query["operator"]);
                    }
                    $shouldInner[] = [
                        'match' => [
                            $name => $query,
                        ],
                    ];
                    $highlight[$name] = new \stdClass();
                }
            }
        }
        if (count($shouldInner) > 0) {
            $should = [
                'nested' => [
                    'path' => 'children',
                    "query" => [
                        'bool' => [
                            'should' => $shouldInner,
                        ],
                    ],
                    'inner_hits' => [
                        'highlight' => [
                            'fields' => $highlight,
                        ],
                    ],
                ],
            ];
            if (array_key_exists('fragmentsize', $config)) {
                $should['nested']['inner_hits']['highlight']['fragment_size'] = $config['fragmentsize'];
            }
            $boolkey = array_key_first($params['body']);
            $bool = ($params['body'][$boolkey]);
            $firstKey = array_key_first($bool);
            $secondKey = array_key_first($bool[$firstKey]);
            if (is_array($bool[$firstKey][$secondKey]) && $kindfilter === false) {
                $params['body']['query'][$firstKey][$secondKey][] = $should;
            }
            if (is_array($bool[$firstKey][$secondKey]) && ($kindfilter)) {
                $first = array_key_first($params['body']['query'][$firstKey][$secondKey]);
                $second = array_key_first($params['body']['query'][$firstKey][$secondKey][$first]);
                $three = array_key_first($params['body']['query'][$firstKey][$secondKey][$first][$second]);
                $params['body']['query']['bool']['must'][$first][$second][$three][] = $should;
            }
        }
    }

    /**
     * Sets the cover and URL for the given values.
     *
     * This method iterates through the search results and sets the cover image and
     * URL for each product based on the variant information found in the inner hits.
     *
     * @param array $values The values array to modify.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function setCoverAndUrl(array &$values): void
    {
        $productId = null;
        $cover = null;
        $merker = [];
        $price = null;
        foreach ($values['hits']['hits'] as &$value) {
            if (is_array($value)) {
                $productId = null;
                $cover = null;
                if (array_key_exists('inner_hits', $value)) {
                    $hits = $value['inner_hits']['children']['hits']['hits'];
                    $firsthit = array_shift($hits);
                    if ($firsthit !== null) {
                        if (array_key_exists("product_id", $firsthit["_source"])) {
                            $productId = $firsthit["_source"]["product_id"];
                        }
                        if (array_key_exists("productId", $firsthit["_source"])) {
                            $productId = $firsthit["_source"]["productId"];
                        }
                        if (array_key_exists("cover", $firsthit["_source"])) {
                            $cover = $firsthit["_source"]["cover"];
                        }
                        if (array_key_exists("price", $firsthit["_source"])) {
                            $price = $firsthit["_source"]["price"];
                        }
                    }
                    if ($productId !== null && !in_array($productId, $merker)) {
                        $value['_source']['channel']['id'] = $productId;
                        $value['_source']['_id'] = $productId;
                        $value['_source']['id'] = $productId;
                        if ($cover !== null) {
                            $value['_source']['channel']['cover'] = $cover;
                        }
                        if ($price !== null) {
                            $value['_source']['channel']['price'] = $price;
                        }
                        $merker[] = $productId;
                    }
                }
            }
        }
    }

    /**
     * Checks if the condition for variant processing is met.
     *
     * This method determines whether the conditions are right for processing product
     * variants based on the provided configuration.
     *
     * @param array $config The configuration array.
     * @return bool
     */
    public function conditionFunction(array $config): bool
    {

        if (array_key_exists('onlymain', $config) && array_key_exists('addVariants', $config)) {
            if (($config['onlymain'] === 'yes' || $config['onlymain'] === 'stock') && ($config['addVariants'] === 'yes' || $config['addVariants'] === 'individual')) {
                return true;
            }
        }
        return false;
    }
}