<?php

namespace Sisi\Search\Service;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\Uuid\Uuid;
use Doctrine\DBAL\Connection;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\Console\Output\OutputInterface;
use Shopware\Core\System\SalesChannel\SalesChannelEntity;
use Shopware\Core\Content\Product\Aggregate\ProductTranslation\ProductTranslationCollection;
use Shopware\Core\Content\Product\Aggregate\ProductTranslation\ProductTranslationEntity;
use Shopware\Core\Content\Category\Aggregate\CategoryTranslation\CategoryTranslationCollection;
use Shopware\Core\Content\Product\Aggregate\ProductManufacturer\ProductManufacturerCollection;
use Shopware\Core\Content\Product\Aggregate\ProductManufacturerTranslation\ProductManufacturerTranslationCollection;

/**
 * This class provides methods for handling translations in the search context.
 * It allows retrieving translations, checking language IDs, and fetching language information.
 */
class TranslationService
{

    /**
     * Retrieves a translation based on the provided language ID from a collection of translations.
     *
     * @param CategoryTranslationCollection|ProductTranslationCollection|ProductManufacturerCollection|ProductManufacturerTranslationCollection $translation The collection of translations to search within.
     * @param string $languageId The language ID to search for.
     * @param array $config An optional array of configuration parameters.
     * @return mixed|false The translation object if found, false otherwise.
     */
    public function getTranslationfields($translation, string $languageId, array $config = [])
    {
        // ---- Check if a configLanguage is set in the configuration
        if (array_key_exists('configLanguage', $config)) {
            if (!empty($config['configLanguage'])) {
                $languageId = $config['configLanguage'];
            }
        }

        // ---- Check if the translation object is not null
        if ($translation !== null) {
            // ---- Check if the translation object has a getElements method
            if (method_exists($translation, 'getElements')) {
                $translationValues = $translation->getElements();
                // ---- Iterate over the translation values
                foreach ($translationValues as $value) {
                    // ---- Check if the language ID matches the provided language ID (case-insensitive)
                    if (strtoupper($value->getLanguageId()) == strtoupper($languageId)) {
                        return $value;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Checks if a language ID is set in the provided values or parameters, and returns it.
     * If not found, it returns the language ID from the SalesChannelEntity.
     *
     * @param array $values An array of values to check for the language ID.
     * @param SalesChannelEntity $salechannelItem The SalesChannelEntity to retrieve the language ID from.
     * @param array $parameters An optional array of parameters to check for the language ID.
     * @return mixed|string The language ID if found, otherwise the language ID from the SalesChannelEntity.
     */
    public function chechIsSetLanuageId(array $values, SalesChannelEntity $salechannelItem, array $parameters = [])
    {
        // ---- Check if the languageID is set in the parameters
        if (array_key_exists('languageID', $parameters)) {
            return strtolower($parameters['languageID']);
        }

        // ---- Check if the language ID is present in the values array
        if (array_key_exists('hex(id)', $values)) {
            return $values['hex(id)'];
        } else {
            return $salechannelItem->getLanguageId();
        }
    }

    /**
     * Retrieves the language ID based on the provided parameters.
     *
     * @param array $parameters An array of parameters to check for the language.
     * @param Connection $connection The database connection.
     * @param OutputInterface|null $output An optional output interface for logging messages.
     * @param Logger $loggingService The logging service.
     * @return mixed|array The language ID if found, otherwise an array containing the default language ID.
     */
    public function getLanguageId(
        array $parameters,
        $connection,
        $output,
        $loggingService
    ) {
        // ---- Check if the language parameter is set to 'all'
        if (array_key_exists('language', $parameters)) {
            if ($parameters['language'] === 'all') {
                return 'all';
            }
        }

        // ---- Check if the language parameter is set and languageID is not set
        if (array_key_exists('language', $parameters) && !array_key_exists('languageID', $parameters)) {
            $result = $this->getLanguage($connection, $parameters['language']);
            // ---- Check if the language is not found
            if ($result == null) {
                $message = "Language paramter not found";
                if ($output !== null) {
                    $output->writeln($message);
                }
                $loggingService->log('100', $message);
            }
            return $result;
        }
        return [1];
    }

    /**
     * Retrieves a language from the database based on its name.
     *
     * @param Connection $connection The database connection.
     * @param string $langageName The name of the language to retrieve.
     * @return mixed An array containing the language ID and name, or false if not found.
     */
    private function getLanguage(Connection $connection, string $langageName)
    {
        $query = $connection->createQueryBuilder()
            ->select(['hex(id),name'])
            ->from('language')
            ->where('language.name =:name')
            ->setParameter(':name', $langageName);
        $result = $query->execute();
        return $result->fetch();
    }
}