<?php

namespace Sisi\Search\Service;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\Uuid\Uuid;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * This class provides utility functions for text manipulation and output.
 * It includes methods for writing messages to the console and processing command-line options.
 */
class TextService
{
    /**
     * Writes a message to the console output.
     *
     * @param OutputInterface|null $output The output interface to write to.
     * @param string $message The message to write.
     */
    public function write($output, $message)
    {
        if ($output !== null) {
            $output->writeln($message);
        }
    }

    /**
     * Extracts specific options from an array of options.
     *
     * @param array $options The array of options to process.
     * @return array An array containing the extracted options.
     */
    public function stripOption(array $options): array
    {
        $return = [];
        foreach ($options as &$option) {
            // ---- Extract 'time' option
            $this->_helpCondition($option, 'time', $return);
            // ---- Extract 'step' option
            $this->_helpCondition($option, 'step', $return);
            // ---- Extract 'update' option
            $this->_helpCondition($option, 'update', $return);
            // ---- Extract 'limit' option
            $this->_helpCondition($option, 'limit', $return);
            // ---- Extract 'offset' option
            $this->_helpCondition($option, 'offset', $return);
            // ---- Extract 'main' option
            $this->_helpCondition($option, 'main', $return);
            // ---- Extract 'language' option
            $this->_helpCondition($option, 'language', $return);
            // ---- Extract 'languageID' option
            $this->_helpCondition($option, 'languageID', $return);
            // ---- Extract 'backend' option
            $this->_helpCondition($option, 'backend', $return);
        }
        return $return;
    }

    /**
     * Helper function to conditionally extract an option.
     *
     * @param string|null $option The option string to check.
     * @param string|null $index The index to use for the extracted option.
     * @param array $return The array to store the extracted option in.
     */
    private function _helpCondition($option, $index, &$return): void
    {
        if ($this->_searchOption($option, $index)) {
            $return[$index] = $option;
        }
    }

    /**
     * Extracts specific options (language, languageID, all, shop, delete) from an array of options.
     *
     * @param array $options The array of options to process.
     * @return array An array containing the extracted options.
     */
    public function stripOption2(array $options): array
    {
        $return = [];
        foreach ($options as &$option) {
            // ---- Extract 'language' option
            if ($this->_searchOption($option, 'language')) {
                $return['language'] = $option;
            }
            // ---- Extract 'languageID' option
            if ($this->_searchOption($option, 'languageID')) {
                $return['languageID'] = $option;
            }

            // ---- Extract 'all' option
            if ($this->_searchOption($option, 'all')) {
                $return['all'] = $option;
            }

            // ---- Extract 'shop' option
            if ($this->_searchOption($option, 'shop')) {
                $return['shop'] = $option;
            }
            // ---- Extract 'delete' option
            if ($this->_searchOption($option, 'delete')) {
                $return['delete'] = $option;
            }
        }
        return $return;
    }

    /**
     * Searches for a specific string within an option and extracts the value.
     *
     * @param string|null $option The option string to search in.
     * @param string|null $string The string to search for.
     * @return bool True if the string is found, false otherwise.
     */
    private function _searchOption(&$option, string $string): bool
    {
        $pos = strpos($option, $string);
        if ($pos !== false) {
            $option = str_replace($string . "=", "", $option);
            return true;
        }
        return false;
    }
}