<?php

namespace Sisi\Search\Service;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Sisi\Search\Core\Content\Task\Bundle\DBSchedularEntity;

/**
 * Service class for managing tasks related to Sisi Search.
 * This class provides methods to check task execution logic and retrieve active tasks.
 */
class TaskService
{
    /**
     * Checks if a task's next execution time has arrived and updates the task's execution times accordingly.
     *
     * @param DBSchedularEntity $result The task entity to check.
     * @param EntityRepository $repository The repository for updating the task entity.
     * @param Context $context The context for database operations.
     *
     * @return bool True if the task should be executed, false otherwise.
     */
    public function ifLogik($result, $repository, $context): bool
    {
        // ---- Get the next execution time from the task entity.
        $nextTime = $result->getNextExecutionTime();
        $now = new \DateTime("now");

        // ---- Check if the next execution time is set.
        if ($nextTime !== null) {
            // ---- Compare the next execution time with the current time.
            if ($nextTime->getTimestamp() <= $now->getTimestamp()) {
                // ---- Calculate the next execution time based on the task's time interval.
                $nextTimesec = $now->getTimestamp() + $result->getTime();
                $next = new \DateTime();
                $next->setTimestamp($nextTimesec);

                // ---- Update the task entity with the new execution times.
                $repository->update([
                                        [
                                            'id' => $result->getId(),
                                            'lastExecutionTime' => $now,
                                            'nextExecutionTime' => $next
                                        ]
                                    ], $context);
                return true;
            }
        }
        return false;
    }

    /**
     * Retrieves all active Sisi Search tasks from the database.
     *
     * @param EntityRepository $repository The repository for querying task entities.
     * @param Context $context The context for database operations.
     *
     * @return array An array of active task entities.
     */
    public function addAllSisiTask($repository, $context): array
    {
        // ---- Create a criteria to filter tasks by their 'aktive' status.
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('aktive', 'yes'));

        // ---- Search for tasks matching the criteria and return them as an array.
        return $repository->search($criteria, $context)->getEntities()->getElements();
    }
}