<?php

namespace Sisi\Search\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Sisi\Search\Constants\ConfigKeyConstants;
use Sisi\Search\ESindexing\ProductDataIndexer;
use Sisi\Search\ServicesInterfaces\InterfaceSearchCategorieService;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Shopware\Core\Checkout\Cart\Price\QuantityPriceCalculator;
use Shopware\Core\System\SalesChannel\Context\AbstractSalesChannelContextFactory;
use Topdata\TopdataQueueHelperSW6\Util\UtilDebug;

/**
 * This class is responsible for starting the indexing process.
 * It orchestrates the indexing of products and categories.
 */
class StartService
{
    /**
     * @param SystemConfigService $systemConfigService
     * @param ProductDataIndexer $productDataIndexer
     * @param Connection $connection
     * @param ContainerInterface $container
     * @param QuantityPriceCalculator $priceCalculator
     * @param AbstractSalesChannelContextFactory $salesChannelContextFactory
     * @param Logger $logger
     * @param array $paramters
     * @param OutputInterface | null $output
     * @param InterfaceSearchCategorieService $searchCategorieService
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     **/
    public function startTheIndexing(
        SystemConfigService                $systemConfigService,
        ProductDataIndexer                 $productDataIndexer,
        Connection                         $connection,
        ContainerInterface                 $container,
        QuantityPriceCalculator            $priceCalculator,
        AbstractSalesChannelContextFactory $salesChannelContextFactory,
        Logger                             $logger,
        array                              $paramters,
        ?OutputInterface                   $output,
        InterfaceSearchCategorieService    $searchCategorieService
    ): void
    {
        $str = true;
        $index = 0;
        $count = 0;
        $texthaendler = new TextService();
        $heandler = new CategorieIndexService();
        $insertTime = 0;

        // ---- Check if offset exists in parameters and calculate index and count
        if (array_key_exists('offset', $paramters)) {
            $index = (int)$paramters['offset'] / $paramters['limit'];
            $count = (int)$paramters['offset'];
        }

        while ($str) {
            // ---- Set update parameter if index is not 0
            if ($index != 0) {
                if (!array_key_exists('update', $paramters)) {
                    $paramters['update'] = "1";
                }
            }

            $paramters['offset'] = $index * $paramters['limit'];
            $paramters['backend'] = "1";
            $paramters['counter'] = $index;

            // ---- Populate the index with product data
            $returnValue = $productDataIndexer->populate(
                $connection,
                $container,
                $systemConfigService,
                $priceCalculator,
                $salesChannelContextFactory,
                $logger,
                $paramters,
                $output
            );
            $total = $returnValue['total'];
            $configExtra = $systemConfigService->get("SisiSearch.config");

            // ---- Start category indexing if configured and on the first iteration
            if (array_key_exists(ConfigKeyConstants::CATEGORIEN, $configExtra) && $index == 0) {
                if ($configExtra[ConfigKeyConstants::CATEGORIEN] === "6"
                    || $configExtra[ConfigKeyConstants::CATEGORIEN] === "7"
                    || $configExtra[ConfigKeyConstants::CATEGORIEN] === "8"
                ) {
                    $heandler->startIndex($container, $paramters, $connection, $systemConfigService, $output, $logger, $searchCategorieService);
                }
            }

            if ($returnValue['usetime'] > 0) {
                $insertTime = $returnValue['usetime'];
            }

            $count += $total;
            $texthaendler->write($output, "Now " . number_format($count) . " products in the Index");
            $index++;

            if ($output == null) {
                echo "The next " . number_format($count) . " articles are now being indexed  \n";
            }

            // ---- Break the loop if no more products are found
            if ($total <= 0) {
                $str = false;
            }

            clearstatcache();
            gc_collect_cycles();
        }

        $this->setFinishFlag($insertTime, $connection);
        $texthaendler->write($output, "The index process is finish with " . $count . " products");
    }

    /**
     * Sets the finish flag in the database after the indexing process is complete.
     *
     * @param int $insertTime The time taken for the indexing process.
     * @param Connection $connection The database connection.
     */
    private function setFinishFlag(int $insertTime, Connection $connection): void
    {
        if ($insertTime > 0) {
            $sql = "UPDATE `s_plugin_sisi_search_es_index`
            SET
              `isfinish` = :isfinish
              WHERE time = :time";
            $connection->executeStatement(
                $sql,
                [
                    'isfinish' => 1,
                    'time'     => $insertTime
                ]
            );
        }
    }
}