<?php

declare(strict_types=1);

namespace Sisi\Search\Service;

use Shopware\Core\Checkout\Cart\Price\QuantityPriceCalculator;
use Shopware\Core\Checkout\Cart\Price\Struct\CartPrice;
use Shopware\Core\Checkout\Cart\Price\Struct\PriceCollection as CalculatedPriceCollection;
use Shopware\Core\Checkout\Cart\Price\Struct\QuantityPriceDefinition;
use Shopware\Core\Checkout\Cart\Price\Struct\ReferencePriceDefinition;
use Shopware\Core\Content\Product\Aggregate\ProductPrice\ProductPriceCollection;
use Shopware\Core\Content\Product\DataAbstractionLayer\CheapestPrice\CalculatedCheapestPrice;
use Shopware\Core\Content\Product\DataAbstractionLayer\CheapestPrice\CheapestPrice;
use Shopware\Core\Content\Product\SalesChannel\Price\ReferencePriceDto;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Entity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Pricing\Price;
use Shopware\Core\Framework\DataAbstractionLayer\Pricing\PriceCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Plugin\Exception\DecorationPatternException;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\Unit\UnitCollection;
use Symfony\Contracts\Service\ResetInterface;

/**
 * This class is responsible for calculating product prices, including regular prices, advanced prices based on rules,
 * and the cheapest price. It uses the QuantityPriceCalculator to perform the actual calculations
 * and considers tax rules, currency, and reference prices.
 *
 * @SuppressWarnings(PHPMD.StaticAccess)
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class SisiProductPriceCalculator implements ResetInterface
{
    private EntityRepositoryInterface $unitRepository;

    private QuantityPriceCalculator $calculator;

    private ?UnitCollection $units = null;

    /**
     * @internal
     */
    public function __construct(EntityRepositoryInterface $unitRepository, QuantityPriceCalculator $calculator)
    {
        $this->unitRepository = $unitRepository;
        $this->calculator = $calculator;
    }

    /**
     * Resets the internal unit collection.
     * This is used to clear the cached unit collection, forcing it to be reloaded on the next request.
     */
    public function reset(): void
    {
        $this->units = null;
    }

    /**
     * Calculates the main price for a product based on the provided context and units.
     *
     * @param Entity $product The product entity for which to calculate the price.
     * @param SalesChannelContext $context The sales channel context.
     * @param UnitCollection $units The collection of available units.
     */
    public function calculatePrice(Entity &$product, SalesChannelContext $context, UnitCollection $units): void
    {
        $price = $product->get('price');
        $taxId = $product->get('taxId');
        if ($price === null || $taxId === null) {
            return;
        }
        $reference = ReferencePriceDto::createFromEntity($product);

        $definition = $this->buildDefinition($product, $price, $context, $units, $reference);

        // ---- Calculate the price using the quantity price calculator
        $price = $this->calculator->calculate($definition, $context);

        // ---- Assign the calculated price to the product entity
        $product->assign([
                             'calculatedPrice' => $price,
                         ]);
    }

    /**
     * Calculates advanced prices for a product based on defined rules and quantities.
     *
     * @param Entity $product The product entity.
     * @param SalesChannelContext $context The sales channel context.
     * @param UnitCollection $units The collection of available units.
     */
    public function calculateAdvancePrices(Entity $product, SalesChannelContext $context, UnitCollection $units): void
    {
        $prices = $product->get('prices');
        if ($prices === null) {
            return;
        }

        if (!$prices instanceof ProductPriceCollection) {
            return;
        }

        // ---- Filter prices based on the current sales channel context rules
        $prices = $this->filterRulePrices($prices, $context);
        if ($prices === null) {
            $product->assign(['calculatedPrices' => new CalculatedPriceCollection()]);

            return;
        }

        $prices->sortByQuantity();

        $reference = ReferencePriceDto::createFromEntity($product);

        $calculated = new CalculatedPriceCollection();
        foreach ($prices as $price) {
            $quantity = $price->getQuantityEnd() ?? $price->getQuantityStart();

            $definition = $this->buildDefinition($product, $price->getPrice(), $context, $units, $reference, $quantity);

            $calculated->add($this->calculator->calculate($definition, $context));
        }

        $product->assign(['calculatedPrices' => $calculated]);
    }

    /**
     * Calculates the cheapest price for a product.
     *
     * This method calculates the cheapest price based on the product's price and any existing cheapest price information.
     *
     * @param Entity $product The product entity.
     * @param SalesChannelContext $context The sales channel context.
     * @param UnitCollection $units The collection of available units.
     */
    public function calculateCheapestPrice(&$product, $context, $units)
    {
        $cheapest = $product->get('cheapestPrice');

        if ($cheapest instanceof CheapestPrice) {
            $price = $product->get('price');
            if ($price === null) {
                return;
            }

            $reference = ReferencePriceDto::createFromEntity($product);

            $definition = $this->buildDefinition($product, $price, $context, $units, $reference);

            $calculated = CalculatedCheapestPrice::createFrom(
                $this->calculator->calculate($definition, $context)
            );

            $prices = $product->get('calculatedPrices');

            $hasRange = $prices instanceof CalculatedPriceCollection && $prices->count() > 1;

            $calculated->setHasRange($hasRange);

            $product->assign(['calculatedCheapestPrice' => $calculated]);
        }

        $reference = ReferencePriceDto::createFromCheapestPrice($cheapest);

        $definition = $this->buildDefinition($product, $cheapest->getPrice(), $context, $units, $reference);

        $calculated = CalculatedCheapestPrice::createFrom(
            $this->calculator->calculate($definition, $context)
        );

        $calculated->setHasRange($cheapest->hasRange());
    }

    /**
     * Builds a QuantityPriceDefinition based on the provided parameters.
     *
     * @param Entity $product The product entity.
     * @param PriceCollection $prices The collection of prices.
     * @param SalesChannelContext $context The sales channel context.
     * @param UnitCollection $units The collection of available units.
     * @param ReferencePriceDto $reference The reference price data.
     * @param int $quantity The quantity for the price definition.
     * @return QuantityPriceDefinition
     */
    private function buildDefinition(
        Entity $product,
        PriceCollection $prices,
        SalesChannelContext $context,
        UnitCollection $units,
        ReferencePriceDto $reference,
        int $quantity = 1
    ): QuantityPriceDefinition {

        $price = $this->getPriceValue($prices, $context);

        $taxId = $product->get('taxId');
        $definition = new QuantityPriceDefinition($price, $context->buildTaxRules($taxId), $quantity);
        $definition->setReferencePriceDefinition(
            $this->buildReferencePriceDefinition($reference, $units)
        );

        $definition->setListPrice(
            $this->getListPrice($prices, $context)
        );

        if (method_exists($definition, 'setRegulationPrice')) {
            $definition->setRegulationPrice(
                $this->getRegulationPrice($prices, $context)
            );
        }

        return $definition;
    }

    /**
     * Gets the price value based on the sales channel context.
     *
     * @param PriceCollection $price The collection of prices.
     * @param SalesChannelContext $context The sales channel context.
     * @return float
     */
    private function getPriceValue(PriceCollection $price, SalesChannelContext $context): float
    {
        /** @var Price $currency */
        $currency = $price->getCurrencyPrice($context->getCurrencyId());

        $value = $this->getPriceForTaxState($currency, $context);

        if ($currency->getCurrencyId() !== $context->getCurrency()->getId()) {
            $value *= $context->getContext()->getCurrencyFactor();
        }

        return $value;
    }

    /**
     * Gets the price based on the tax state.
     *
     * @param Price $price The price object.
     * @param SalesChannelContext $context The sales channel context.
     * @return float
     */
    private function getPriceForTaxState(Price $price, SalesChannelContext $context): float
    {
        if ($context->getTaxState() === CartPrice::TAX_STATE_GROSS) {
            return $price->getGross();
        }

        return $price->getNet();
    }

    /**
     * Gets the list price from the price collection.
     *
     * @param PriceCollection|null $prices The collection of prices.
     * @param SalesChannelContext $context The sales channel context.
     * @return float|null
     */
    private function getListPrice(?PriceCollection $prices, SalesChannelContext $context): ?float
    {
        if (!$prices) {
            return null;
        }

        $price = $prices->getCurrencyPrice($context->getCurrency()->getId());
        if ($price === null || $price->getListPrice() === null) {
            return null;
        }

        $value = $this->getPriceForTaxState($price->getListPrice(), $context);

        if ($price->getCurrencyId() !== $context->getCurrency()->getId()) {
            $value *= $context->getContext()->getCurrencyFactor();
        }

        return $value;
    }

    /**
     * Gets the regulation price from the price collection.
     *
     * @param PriceCollection|null $prices The collection of prices.
     * @param SalesChannelContext $context The sales channel context.
     * @return float|null
     */
    private function getRegulationPrice(?PriceCollection $prices, SalesChannelContext $context): ?float
    {
        if (!$prices) {
            return null;
        }

        $price = $prices->getCurrencyPrice($context->getCurrency()->getId());
        if ($price === null || $price->getRegulationPrice() === null) {
            return null;
        }

        $taxPrice = $this->getPriceForTaxState($price, $context);
        $value = $this->getPriceForTaxState($price->getRegulationPrice(), $context);
        if ($taxPrice === 0.0 || $taxPrice === $value) {
            return null;
        }

        if ($price->getCurrencyId() !== $context->getCurrency()->getId()) {
            $value *= $context->getContext()->getCurrencyFactor();
        }

        return $value;
    }

    /**
     * Builds a ReferencePriceDefinition based on the provided parameters.
     *
     * @param ReferencePriceDto $definition The reference price data.
     * @param UnitCollection $units The collection of available units.
     * @return ReferencePriceDefinition|null
     */
    private function buildReferencePriceDefinition(
        ReferencePriceDto $definition,
        UnitCollection $units
    ): ?ReferencePriceDefinition {
        if ($definition->getPurchase() === null || $definition->getPurchase() <= 0) {
            return null;
        }
        if ($definition->getUnitId() === null) {
            return null;
        }
        if ($definition->getReference() === null || $definition->getReference() <= 0) {
            return null;
        }
        if ($definition->getPurchase() === $definition->getReference()) {
            return null;
        }

        $unit = $units->get($definition->getUnitId());
        if ($unit === null) {
            return null;
        }

        return new ReferencePriceDefinition(
            $definition->getPurchase(),
            $definition->getReference(),
            $unit->getTranslation('name')
        );
    }

    /**
     * Filters product prices based on the provided rule IDs.
     *
     * @param ProductPriceCollection $rules The collection of product prices.
     * @param SalesChannelContext $context The sales channel context.
     * @return ProductPriceCollection|null
     */
    private function filterRulePrices(
        ProductPriceCollection $rules,
        SalesChannelContext $context
    ): ?ProductPriceCollection {
        foreach ($context->getRuleIds() as $ruleId) {
            $filtered = $rules->filterByRuleId($ruleId);

            if (\count($filtered) > 0) {
                return $filtered;
            }
        }

        return null;
    }

    /**
     * Gets the collection of units.
     *
     * @param SalesChannelContext $context The sales channel context.
     * @return UnitCollection
     */
    public function getUnits(SalesChannelContext $context): UnitCollection
    {
        if ($this->units !== null) {
            return $this->units;
        }

        $criteria = new Criteria();
        $criteria->setTitle('product-price-calculator::units');

        /** @var UnitCollection $units */
        $units = $this->unitRepository
            ->search($criteria, $context->getContext())
            ->getEntities();

        return $this->units = $units;
    }
}