<?php

namespace Sisi\Search\Service;

use Shopware\Core\Content\Product\Aggregate\ProductSearchKeyword\ProductSearchKeywordCollection;
use Shopware\Core\Content\Product\Aggregate\ProductSearchKeyword\ProductSearchKeywordEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepositoryInterface;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Sisi\Search\Constants\ConfigKeyConstants;
use Sisi\Search\Util\CliLogger;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\Console\Output\OutputInterface;
use Topdata\TopdataQueueHelperSW6\Util\UtilDebug;

/**
 * This class provides services for managing and inserting search keywords for products.
 * It handles custom search keywords and merging of existing keywords.
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class SearchkeyService
{
    /**
     * Inserts search keywords for a given product entity.
     *
     * This method retrieves search keywords from the product entity, processes them,
     * and inserts them into the search index. This is now controlled by the `useShopwareSearchKeywords` config.
     * It also handles custom search keywords and adds product numbers or manufacturer numbers as keywords based on configuration.
     *
     * @param array $fields The fields to be indexed. This is passed by reference and modified.
     * @param mixed $productEntity The product entity.
     * @param string $lanuageId The language ID for the keywords.
     * @param EntitySearchResult $mappingValues The mapping values for the fields.
     * @param array $config The configuration array.
     * @param Logger $logger The logging service.
     * @param string|null $parentid The parent ID of the product, if it's a variant.
     * @param InsertService $insertService The InsertService instance.
     * @param mixed $connection The database connection.
     *
     * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
     * @SuppressWarnings(PHPMD.CyclomaticComplexity )
     */
    public function insertSearchKey(
        &$fields,
        $productEntity,
        $lanuageId,
        &$mappingValues,
        $config,
        $logger,
        $parentid,
        $insertService,
        $connection
    )
    {
        // CliLogger::debug("Starting search keyword processing for product: " . $productEntity->getId());

        $collectedSearchKeywordEntities = []; // List of ProductSearchKeywordEntity
        $collectedKeywordStrings = [];    // List of keyword strings

        // --- Part 1: Conditionally get native Shopware search keywords ---
        if (isset($config[ConfigKeyConstants::USE_SHOPWARE_SEARCH_KEYWORDS]) && $config[ConfigKeyConstants::USE_SHOPWARE_SEARCH_KEYWORDS] === 'yes') {
            // CliLogger::info("Checking for native Shopware search keywords");
            $nativeSearchKeywords = $productEntity->getSearchKeywords();
            if ($nativeSearchKeywords !== null) {
                // CliLogger::info("Processing " . count($nativeSearchKeywords) . " native keywords");
                foreach ($nativeSearchKeywords as $woert) {
                    $collectedSearchKeywordEntities[] = $woert;
                    if (strtoupper($woert->getLanguageId()) === strtoupper($lanuageId)) {
                        $collectedKeywordStrings[] = $woert->getKeyword();
                    }
                }
            } else {
                // CliLogger::debug("No native search keywords found for product");
            }
        } else {
            // use the custom search keywords (mcx hack 2025-06-20)
            if(is_array($productEntity->getCustomSearchKeywords())) {
                foreach ($productEntity->getCustomSearchKeywords() as $customKeyword) {
                    $collectedKeywordStrings[] = $customKeyword;
                    $newKeywordEntity = new ProductSearchKeywordEntity();
                    $newKeywordEntity->setKeyword($customKeyword);
                    $newKeywordEntity->setProductId($productEntity->getId());
                    $collectedSearchKeywordEntities[] = $newKeywordEntity;
                }
            }
        }

        // --- Part 2: Always check for adding variant product/manufacturer numbers ---
        if (array_key_exists(ConfigKeyConstants::ADD_PRODUCT_NUMBER, $config) && $config[ConfigKeyConstants::ADD_PRODUCT_NUMBER] === 'yes') {
            // CliLogger::info("Adding product/manufacturer numbers as keywords");
            $extendInsertService = new ExtendInsertService();
            if ($productEntity->getParentId() === null) {
                $variantDataArray = $extendInsertService->dbqueryfromStockFromAllVaraints($connection, $productEntity->getId());
                if (count($variantDataArray) > 0) {
                    foreach ($variantDataArray as $item) {
                        // Add product number as keyword
                        if (!empty($item['product_number'])) {
                            // CliLogger::debug("Adding product number: " . $item['product_number']);
                            $collectedKeywordStrings[] = $item['product_number'];
                            $newKeywordEntity = new ProductSearchKeywordEntity();
                            $newKeywordEntity->setKeyword($item['product_number']);
                            $newKeywordEntity->setProductId($item['HEX(id)']);
                            $collectedSearchKeywordEntities[] = $newKeywordEntity;
                        }
                        // Add manufacturer number as keyword
                        if (!empty($item['manufacturer_number'])) {
                            // CliLogger::debug("Adding manufacturer number: " . $item['manufacturer_number']);
                            $collectedKeywordStrings[] = $item['manufacturer_number'];
                            $newKeywordEntity = new ProductSearchKeywordEntity();
                            $newKeywordEntity->setKeyword($item['manufacturer_number']);
                            $newKeywordEntity->setProductId($item['HEX(id)']);
                            $collectedSearchKeywordEntities[] = $newKeywordEntity;
                        }
                    }
                } else {
                    // CliLogger::debug("No variant data found for product");
                }
            } else {
                // CliLogger::debug("Skipping product number addition for variant product");
            }
        }


        // --- Part 3: If we have any keywords, process them ---
        if (!empty($collectedSearchKeywordEntities)) {
            // CliLogger::debug("Collected keywords: " . implode(', ', $collectedKeywordStrings));

//            UtilDebug::dd(
//                $collectedKeywordStrings,
//                count($collectedKeywordStrings),
//                count($collectedSearchKeywordEntities)
//            );

            // The `checkFunction` requires a single entity instance for its checks,
            // even though it processes the full array from the `ext` parameter.
            $firstKeywordEntity = $collectedSearchKeywordEntities[0];

            $insertService->checkFunction(
                $mappingValues,
                $fields,
                $firstKeywordEntity,
                $collectedKeywordStrings, // This argument is not effectively used but required by signature
                $logger,
                $config,
                'search',
                true,
                $parentid,
                ['multivalue' => $collectedSearchKeywordEntities] // This is where the actual data is read from
            );
        }

        // --- Part 4: Always insert custom search keywords ---
        // CliLogger::debug("Inserting custom search keywords");
        $this->insertCustomSearchKey($config, $productEntity, $fields);
    }

    /**
     * Inserts custom search keywords for a given product entity.
     *
     * This method retrieves custom search keywords from the product entity, processes them,
     * and adds them to the fields to be indexed.
     *
     * @param array $config The configuration array.
     * @param mixed $entity The product entity.
     * @param array $fields The fields to be indexed. This is passed by reference and modified.
     */
    public function insertCustomSearchKey(array $config, $entity, &$fields)
    {
        // CliLogger::debug("Processing custom keywords for product: " . $entity->getId());
        $customSearchKeywords = null;
        $arrayKey = 'product_name';

        // ---- Check if custom search keywords are enabled in the configuration
        if (array_key_exists(ConfigKeyConstants::SEARCHKEYWORT, $config)) {
            if ($config[ConfigKeyConstants::SEARCHKEYWORT] === '3') {
//                // CliLogger::info("Custom keyword processing enabled");
                $customSearchKeywords = $entity->getCustomSearchKeywords();
//               UtilDebug::dd($customSearchKeywords, $fields);
            } else {
//                // CliLogger::debug("Custom keyword processing disabled (searchkeywort: " . $config[ConfigKeyConstants::SEARCHKEYWORT] . ")");
            }
        } else {
//            // CliLogger::debug("Custom keyword processing not configured");
        }

        // ---- Check if a prefix for the search keyword field is defined in the configuration
        if (array_key_exists(ConfigKeyConstants::PREFIXSEARCHKEYWORT, $config)) {
            if (!empty($config[ConfigKeyConstants::PREFIXSEARCHKEYWORT])) {
                // CliLogger::debug("Using prefix for search keyword field: " . $config[ConfigKeyConstants::PREFIXSEARCHKEYWORT]);
                $arrayKey = trim($config[ConfigKeyConstants::PREFIXSEARCHKEYWORT]) . $arrayKey;
            }
        }

        // ---- If custom search keywords are available, add them to the field
        if ($customSearchKeywords !== null) {
//             CliLogger::info("Adding custom keywords: " . implode(', ', $customSearchKeywords));
            if (array_key_exists($arrayKey, $fields)) {
//                $customKey = "";
//                foreach ($customSearchKeywords as $keyword) {
//                    $customKey .= " " . $keyword;
//                }
//                $fields[$arrayKey] .= $customKey;
                $fields[$arrayKey] .= ' ' . implode(' ', $customSearchKeywords);
            } else {
                // CliLogger::warning("Field key '" . $arrayKey . "' not found in fields array");
            }
        } else {
            // CliLogger::debug("No custom keywords to add");
        }

        // ---- Debug
//        UtilDebug::dd($keywords, $fields);
    }

    public function addFilter(array &$return, array $config, array $synoms): void
    {
        if (array_key_exists(ConfigKeyConstants::SEARCHKEYWORT, $config) && (count($synoms) > 0)) {
            if ($config[ConfigKeyConstants::SEARCHKEYWORT] === '1') {
                $return["analysis"]["filter"]['product_shopwarekeywords']["type"] = "synonym";
                $return["analysis"]["filter"]['product_shopwarekeywords']["synonyms"] = $synoms;
            }
            if (array_key_exists('analyzer_product_name', $return["analysis"]["analyzer"])) {
                $return["analysis"]["analyzer"]["analyzer_product_name"]['filter'][] = "product_shopwarekeywords";
            }
        }
    }

    /**
     * Merges search keywords from products to create synonyms.
     *
     * This method retrieves product search keywords and merges them to create a list of synonyms
     * for use in search analysis.
     *
     * @param array $config The configuration array.
     * @param SalesChannelRepository | null $productService The product service.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param array $paramters The parameters array.
     * @param OutputInterface|null $output The output interface.
     * @return array The array of synonyms.
     */
    public function mergeSearchkeywoerter(
        $config,
        $productService,
        $saleschannelContext,
        $paramters,
        $output
    ): array
    {
        // CliLogger::info("Starting keyword merging for language: " . $paramters['language_id']);
        $return = [];

        // ---- Check if search keyword merging is enabled in the configuration and if it's the first iteration
        if (array_key_exists(ConfigKeyConstants::SEARCHKEYWORT, $config) && $paramters['counter'] === 0) {
            $texthaendler = new TextService();

            // ---- If the configuration specifies to merge Shopware keywords
            if ($config[ConfigKeyConstants::SEARCHKEYWORT] === '1') {
                $texthaendler->write($output, "First we will merge the Shopware keywords.\n");
                $criteria = new Criteria();
                $criteria->addAssociation('searchKeywords');
                $criteria->addAssociation('translations');

                // ---- Set the limit if specified in the parameters
                if (array_key_exists('limit', $paramters)) {
                    $criteria->setlimit($paramters['limit']);
                }

                $lanuageId = $paramters['language_id'];
                $offset = 0;

                // ---- Get the synonyms in steps
                // CliLogger::debug("Retrieving synonyms with offset: " . $offset);
                $this->getSynomeBysteps(
                    $criteria,
                    $productService,
                    $saleschannelContext,
                    $lanuageId,
                    $return,
                    $paramters,
                    $offset
                );
            }
        }
        return $return;
    }

    /**
     * Extends the merged keywords with additional keywords.
     *
     * @param ProductSearchKeywordCollection $searchkeywoerter The collection of search keywords.
     * @param string $lanuageId The language ID.
     * @param int $index The index of the current keyword.
     * @param array $return The array to store the merged keywords.
     */
    private function extendMergekeywords(
        ProductSearchKeywordCollection $searchkeywoerter,
        string                         $lanuageId,
        int                            &$index,
        array                          &$return
    ): void
    {
        foreach ($searchkeywoerter as $woert) {
            if (
                strtoupper($woert->getLanguageId()) === strtoupper($lanuageId)
                && !empty($woert->getKeyword())
            ) {
                // CliLogger::debug("Adding keyword: " . $woert->getKeyword());
                $return[$index] .= "," . $woert->getKeyword();
            }
        }
    }

    /**
     * Retrieves synonyms in steps using a criteria and offset.
     *
     * This method iterates through products and their search keywords to build a list of synonyms.
     *
     * @param Criteria $criteria The criteria for the product search.
     * @param SalesChannelRepositoryInterface $productService The product service.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param string $lanuageId The language ID.
     * @param array $return The array to store the synonyms.
     * @param array $paramters The parameters array.
     * @param int $offset The offset for the product search.
     */
    private function getSynomeBysteps(
        Criteria                        &$criteria,
        SalesChannelRepositoryInterface $productService,
        SalesChannelContext             $saleschannelContext,
        string                          $lanuageId,
        array                           &$return,
        array                           $paramters,
        int                             $offset
    ): void
    {
        // CliLogger::debug("Getting synonyms in steps with offset: " . $offset);
        $criteria->setOffset($offset);
        $str = true;
        $index = 0;

        // ---- Iterate through the products until no more results are found
        while ($str) {
            // CliLogger::debug("Iterating for synonyms with offset: " . $offset);
            $result = $this->iterateSynoms(
                $saleschannelContext,
                $criteria,
                $productService,
                $lanuageId,
                $index,
                $return
            );

            // ---- Increment the offset based on the limit or a default value
            if (array_key_exists('limit', $paramters)) {
                $offset = $offset + $paramters['limit'];
            } else {
                $offset = $offset + 1000;
            }
            // CliLogger::debug("Updating offset to: " . $offset);
            $criteria->setOffset($offset);

            // ---- If no more results are found, stop iterating
            if (count($result) === 0) {
                // CliLogger::debug("No more results found, stopping iteration");
                $str = false;
            }
        }
    }

    /**
     * Iterates through the products and extracts synonyms.
     *
     * This method retrieves products based on the criteria and extracts their search keywords
     * to build a list of synonyms.
     *
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param Criteria $criteria The criteria for the product search.
     * @param SalesChannelRepositoryInterface $productService The product service.
     * @param string $lanuageId The language ID.
     * @param string $index The index for the synonyms.
     * @param array $return The array to store the synonyms.
     * @return EntitySearchResult The result of the product search.
     */
    private function iterateSynoms(
        SalesChannelContext             $saleschannelContext,
        Criteria                        &$criteria,
        SalesChannelRepositoryInterface $productService,
        string                          $lanuageId,
        string                          &$index,
        array                           &$return
    ): EntitySearchResult
    {
        // CliLogger::debug("Iterating through products for synonyms");
        /** @var EntitySearchResult<SalesChannelProductEntity> $entities */
        $entities = $productService->search($criteria, $saleschannelContext);
        if ($entities !== false) {
            foreach ($entities as $entity) {
                $searchkeywoerter = $entity->getSearchKeywords();
                if ($searchkeywoerter !== null) {
                    $translation = $entity->getTranslations();
                    foreach ($translation->getElements() as $translationItem) {
                        if ($translationItem->getlanguageId() === $lanuageId) {
                            // CliLogger::debug("Processing product: " . $entity->getId());
                            $return[$index] = '';
                            $this->extendMergekeywords($searchkeywoerter, $lanuageId, $index, $return);
                            if (!empty($return[$index])) {
                                $return[$index] = $translationItem->getName() . '' . $return[$index];
                                // CliLogger::debug("Generated synonym: " . $return[$index]);
                            }
                        }
                    }
                    $index++;
                }
            }
        } else {
            // CliLogger::warning("No entities found for criteria");
        }
        return $entities;
    }
}