<?php

namespace Sisi\Search\Service;

use Elasticsearch\Client;
use Shopware\Core\Content\Category\CategoryEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Sisi\Search\Core\Content\Fields\Bundle\DBFieldsEntity;
use Sisi\Search\ServicesInterfaces\InterfaceSearchCategorieService;

/**
 * Service class for searching categories using Elasticsearch.
 *
 * This class provides methods to search categories based on different configurations
 * and to manage category data within Elasticsearch indices.
 *
 * @SuppressWarnings("ExcessiveClassComplexity")
 */
class SearchCategorieService implements InterfaceSearchCategorieService
{
    /**
     * Searches for categories based on the provided configuration and term within the main index.
     *
     * @param array $config Configuration array.
     * @param array $params Parameters array.
     * @param array $configkategorie Category configuration array.
     * @param client $client Elasticsearch client.
     * @param string $term Search term.
     * @return array|void Returns the search results or void if the category search is disabled.
     */
    public function searchCategorie($config, $params, $configkategorie, $client, $term)
    {
        if (array_key_exists('categorien', $config)) {
            if ($config['categorien'] !== '2') {
                // ---- Prepare parameters for the Elasticsearch query
                $paramsForPropertie['index'] = $params['index'];
                $paramsForPropertie['body']['query'] = '';

                // ---- Build the 'should' array for the nested query
                foreach ($configkategorie as $queries) {
                    if (array_key_exists('match', $queries)) {
                        foreach ($queries['match'] as $key => $item) {
                            $should[]["match"]['categories.' . $key] = $item;
                        }
                    }
                }

                $should[] = [
                    'match' => [
                        'categories.category_id' =>  $term,

                    ]
                ];

                // ---- Construct the Elasticsearch query
                $paramsForPropertie = [
                    'index' => $params['index'],
                    'body' => [
                        'query' => [
                            'nested' => [
                                "path" => "categories",
                                'query' => [
                                    'bool' => [
                                        'should' => $should
                                    ]
                                ],
                                'inner_hits' => [
                                    'highlight' => [
                                        'pre_tags' => ["<b>"], // not required
                                        'post_tags' => ["</b>"], // not required,
                                        'fields' => [
                                            'categories.category_name' => new \stdClass()
                                        ],
                                        'require_field_match' => false
                                    ]
                                ]
                            ]
                        ],

                    ],
                ];

                // ---- Set fragment size for highlighting if configured
                if (array_key_exists('fragmentsizecategorie', $config)) {
                    $framisize = (int)$config['fragmentsizecategorie'];
                    if (!empty($config['fragmentsizecategorie']) && $framisize > 0) {
                        $paramsForPropertie['body']['query']['nested']['inner_hits']['highlight']['fragment_size'] = $framisize;
                    }
                }

                return $client->search($paramsForPropertie);
            }
        }
    }

    /**
     * Searches for categories based on the provided configuration and term within its own index.
     *
     * @param array $config Configuration array.
     * @param $params
     * @param $configkategorie
     * @param $client
     * @param $term
     * @return array Returns the search results.
     *
     * @SuppressWarnings("unused")
     * @SuppressWarnings("CyclomaticComplexity")
     */
    public function searchCategorieWithOwnIndex($config, $params, $configkategorie, $client, $term)
    {
        if (array_key_exists('categorien', $config)) {
            // ---- Determine the index name
            $index = "categorien_" . $params['index'];
            $index = $this->setPrefix($index, $config);

            if ($config['categorien'] === '6' || $config['categorien'] === '7' || $config['categorien'] === '8') {
                // ---- Build the 'should' array for the query
                foreach ($configkategorie as $queries) {
                    if (array_key_exists('match', $queries)) {
                        foreach ($queries['match'] as $key => $item) {
                            $should[]["match"][$key] = $item;
                        }
                    }
                    if (array_key_exists('match_phrase_prefix', $queries)) {
                        foreach ($queries['match_phrase_prefix'] as $key => $item) {
                            $should[]["match"][$key] = $item;
                        }
                    }
                    if (array_key_exists('match_phrase', $queries)) {
                        foreach ($queries['match_phrase'] as $key => $item) {
                            $should[]["match"][$key] = $item;
                        }
                    }
                }

                $should[] = [
                    'match' => [
                        'category_id' =>  $term,

                    ]
                ];

                // ---- Construct the Elasticsearch query
                $paramsForPropertie = [
                    'index' =>  $index,
                    'body' => [
                        'query' => [
                            'bool' => [
                                'should' => $should
                            ],
                        ],
                        'highlight' => [
                            'pre_tags' => ["<b>"], // not required
                            'post_tags' => ["</b>"], // not required,
                            'fields' => [
                                'category_name' => new \stdClass()
                            ],
                            'require_field_match' => false
                        ]
                    ],
                ];

                // ---- Set fragment size for highlighting if configured
                if (array_key_exists('fragmentsizecategorie', $config)) {
                    $framisize = (int)$config['fragmentsizecategorie'];
                    if (!empty($config['fragmentsizecategorie']) && $framisize > 0) {
                        $paramsForPropertie['body']['highlight']['fragment_size'] = $framisize;
                    }
                }

                return $client->search($paramsForPropertie);
            }
        }
        return [];
    }

    /**
     * Sets the index prefix based on the configuration.
     *
     * @param string $index The index name.
     * @param array $config The configuration array.
     * @return string The index name with the prefix, if configured.
     */
    private function setPrefix(string $index, array $config): string
    {
        if (array_key_exists('prefix', $config) && array_key_exists('useprefixforcategorie', $config)) {
            if ($config['useprefixforcategorie'] === '1') {
                 return $config['prefix'] . $index;
            }
        }
        return $index;
    }

    /**
     * Inserts category data into the Elasticsearch index.
     *
     * @param Client $client Elasticsearch client.
     * @param string $indexname Index name.
     * @param CategoryEntity $category Category entity.
     * @param array $fieldConfig Field configuration.
     * @param array $config Configuration array.
     * @param array $parameter Parameters array.
     * @return array Returns the result of the index operation.
     */
    public function insertValue($client, $indexname, $category, $fieldConfig, $config, $parameter): array
    {
        $fields = [];
        $henadler = new CategorieInsertService();
        $henadler->mergeFields($fieldConfig, $fields, $config, $category, $parameter);
        $type = $category->getType();
        $originCategoryBreadcrumb = $fields["category_breadcrumb"];

        // ---- Remove breadcrumb parts if configured
        if (!empty($config["removeBreadcrumb"])) {
            $value = $config["removeBreadcrumb"];
            $chunks = array_filter(explode("\n", $value));
        
            foreach ($chunks as $chunk) {
                if (strpos($originCategoryBreadcrumb, $chunk) !== false) {
                    $originCategoryBreadcrumb = str_replace($chunk, '', $originCategoryBreadcrumb);
                }
            }
        }
        $fields["category_breadcrumb"] = $originCategoryBreadcrumb;

        // ---- Index the category if it's a 'page' type
        if ($type === "page") {
            $params = [
                'index' => $indexname,
                'id' => strtolower($category->getId()),
                'body' => $fields
            ];
            return $client->index($params);
        } else {
            return [];
        }
    }

    /**
     * Creates a criteria object for category search.
     *
     * @return Criteria Returns the criteria object.
     */
    public function createCriteria(): Criteria
    {
        $criteria = new Criteria();
        $criteria->addAssociation('translations');
        $criteria->addAssociation('children');
        $criteria->addAssociation('products');
        $criteria->addFilter(new EqualsFilter('type', 'page'));

        return  $criteria;
    }

    /**
     * Creates the category mapping for Elasticsearch.
     *
     * @param array $fieldConfig Field configuration array.
     * @return array Returns the mapping array.
     */
    public function createCategoryMapping(array $fieldConfig): array
    {
        foreach ($fieldConfig as $backendconfig) {
            $name = $backendconfig->getPrefix() . $backendconfig->getTablename() . "_" . $backendconfig->getName();
            $analyzer = "analyzer_" . $name;
            $type = $backendconfig->getFieldtype();
            $mapping['properties'][$name] = [
                "type" => $type,
                "analyzer" => $analyzer
            ];
        }

        $mapping['properties']["category_id"] = [
            "type" => "text"
        ];

        $mapping['properties']["category_breadcrumb"] = [
            "type" => "text"
        ];
        return $mapping;
    }

    /**
     * Creates the category settings for Elasticsearch.
     *
     * @param array $fieldConfigs Field configurations.
     * @param array $config Configuration array.
     * @return array Returns the settings array.
     *
     * @SuppressWarnings(PHPMD)
     */
    public function createCategorySettings(array $fieldConfigs, array $config)
    {
        $stopsWords = [];
        $stemmervalues = [];
        $settings = [];

        // ---- Set index settings from config
        if (array_key_exists('maxngramdiff', $config)) {
            if (!empty($config['maxngramdiff'])) {
                $settings['index']['max_ngram_diff'] = $config['maxngramdiff'];
            }
        }
        if (array_key_exists('maxshinglediff', $config)) {
            if (!empty($config['maxshinglediff'])) {
                $settings['index']['max_shingle_diff'] = $config['maxshinglediff'];
            }
        }
        if (array_key_exists('totalfields', $config)) {
            $settings['index']['mapping']['total_fields']['limit'] = $config['totalfields'];
        }

        // ---- Loop through field configurations to create analyzers
        foreach ($fieldConfigs as $key => $backendconfig) {
            $minGram = 3;
            $maxGram = 3;
            $filter = [];
            $tokenizer = "ngram";
            $stemming = "";
            $strstop = "";
            $name = $backendconfig->getPrefix() . $backendconfig->getTablename() . "_" . $backendconfig->getName();

            // ---- Handle stopwords
            if (!empty($backendconfig->getStemmingstop())) {
                $valuestring = str_replace("\n", "", $backendconfig->getStop());
                $stopsWords[$name] = explode(",", $valuestring);
            }
            if (!empty($backendconfig->getStop())) {
                $strstop = $backendconfig->getStop();
            }
            if (!empty($backendconfig->getMinedge())) {
                $minGram = (int)$backendconfig->getMinedge();
            }

            if (!empty($backendconfig->getEdge())) {
                $maxGram = (int)$backendconfig->getEdge();
            }

            $filterstring = '';
            $filterstringName = '';

            // ---- Handle filter 1
            if (!empty($backendconfig->getFilter1()) && $backendconfig->getFilter1() !== 'noselect') {
                 $filterstring = $backendconfig->getFilter1();
            }

            $filterstringName = $filterstringName = $this->mergeFiltername($filterstring, $name);

            $this->setSynonymFilter($filterstring, $filterstringName, $backendconfig, $settings);

            if ($filterstringName !== '') {
                $filter[] = $filterstringName;
            }

            $filterstring = '';
            $filterstringName = '';

            // ---- Handle filter 2
            if (!empty($backendconfig->getFilter2()) && $backendconfig->getFilter2() !== 'noselect') {
                $filterstring = $backendconfig->getFilter2();
            }

            $filterstringName = $filterstringName = $this->mergeFiltername($filterstring, $name);

            $this->setSynonymFilter($filterstring, $filterstringName, $backendconfig, $settings);

            if ($filterstring !== '') {
                $filter[] = $filterstringName;
            }

            $filterstring = '';
            $filterstringName = '';

            $filterstringName = $filterstringName = $this->mergeFiltername($filterstring, $name);

            // ---- Handle filter 3
            if (!empty($backendconfig->getFilter3()) && $backendconfig->getFilter3() !== 'noselect') {
                $filterstring = $backendconfig->getFilter3();
            }

            $this->setSynonymFilter($filterstring, $filterstringName, $backendconfig, $settings);

            if ($filterstring !== '') {
                $filter[] = $filterstringName;
            }

            // ---- Handle tokenizer and stemming
            if (!empty($backendconfig->getTokenizer())) {
                $tokenizer = $backendconfig->getTokenizer();
            }

            if (!empty($backendconfig->getStemming())) {
                $stemming = $backendconfig->getStemming();
            }

            if ($strstop === 'yes') {
                $filter[] = "stop_" . $name;
            }
            if ($tokenizer === "Edgengramtokenizer") {
                $tokenizer = "edge_ngram";
            }

            $tokenChars = ["letter", "digit"];

            if ($backendconfig->getPunctuation() == 'yes') {
                $tokenChars[] = "punctuation";
            }

            if ($backendconfig->getWhitespace() == 'yes') {
                $tokenChars[] = "whitespace";
            }

            // ---- Configure ngram tokenizer
            if ($tokenizer === "edge_ngram" || $tokenizer === "ngram") {
                $settings["analysis"]["tokenizer"][$name . "_" . $tokenizer] = [
                    "token_chars" => $tokenChars,
                    "min_gram" => $minGram,
                    "max_gram" => $maxGram,
                    "type" => $tokenizer
                ];
                $tokenizer = $name . "_" . $tokenizer;
            }

            // ---- Configure stemming filter
            if (!empty($stemming)) {
                $stemmerName = "stemmer_" . $name;
                $filter[] = $stemmerName;
                $stemmervalues[$stemmerName] = $stemming;
            }

            $analyzer = "analyzer_" . $name;
            $settings["analysis"]["analyzer"][$analyzer]["filter"] = $filter;
            $settings["analysis"]["analyzer"][$analyzer]["tokenizer"] = $tokenizer;
        }

        // ---- Configure stopwords filter
        foreach ($stopsWords as $key => $stopsWordsItem) {
            $settings["analysis"]["filter"]["stop_ " . $key] = [
                "type" => "stop",
                "ignore_case" => true,
                "stopwords" => $stopsWordsItem
            ];
            $filter[] = "stop_ " . $key;
        }

        // ---- Configure stemmer filter
        foreach ($stemmervalues as $key => $stemmeritem) {
            $settings["analysis"]["filter"][$key] = [
                "type" => "stemmer",
                "language" => $stemmeritem
            ];
            $filter[] = $key;
        }

        $this->setAutocompetefilter($settings, $config);

        return $settings;
    }

    /**
     * Sets the synonym filter in the Elasticsearch settings.
     *
     * @param string $filterString The filter string.
     * @param string $filterName The filter name.
     * @param DBFieldsEntity $backendconfig The backend configuration.
     * @param array $settings The settings array (passed by reference).
     */
    private function setSynonymFilter(string $filterString, string $filterName, DBFieldsEntity $backendconfig, array &$settings): void
    {
        if ($filterString === 'synonym') {
            $productExtend = new ProductExtendService();
            $values = $productExtend->getSynonymvalue($backendconfig);
            $settings["analysis"]["filter"][$filterName] = [
                "type" => "synonym",
                "synonyms" => $values
            ];
        }
    }

    /**
     * Merges the filter name with the base name.
     *
     * @param string $filterstring The filter string.
     * @param string $name The base name.
     * @return string The merged filter name.
     */
    private function mergeFiltername(string $filterstring, string $name): string
    {
        if ($filterstring  === 'synonym') {
            return $name . "_" . $filterstring;
        }
        return $filterstring;
    }

    /**
     * Sets the autocomplete filter in the Elasticsearch settings.
     *
     * @param array $settings The settings array (passed by reference).
     * @param array $config The configuration array.
     */
    public function setAutocompetefilter(array &$settings, array $config): void
    {
        $min = 3;
        $max = 12;

        if (array_key_exists('minedge', $config)) {
            if (!empty($config['minedge'])) {
                $min = $config['minedge'];
            }
        }
        if (array_key_exists('edge', $config)) {
            if (!empty($config['edge'])) {
                $max = $config['edge'];
            }
        }
        $settings["analysis"]["filter"]['autocomplete'] = [
            "type" => "edge_ngram",
            "min_gram" => $min,
            "max_gram" => $max
        ];
    }
}