<?php

namespace Sisi\Search\Service;

use Elasticsearch\Client;
use MyProject\Container;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionCollection;
use Shopware\Core\Content\Property\PropertyGroupEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Sisi\Search\ServicesInterfaces\InterfaceFrontendService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This class provides services related to product properties,
 * including sorting and querying properties based on configuration.
 */
class PropertiesService
{
    /**
     * Executes a search query using the provided parameters and configuration.
     *
     * @param InterfaceFrontendService $frontendService The frontend service to use for searching.
     * @param Client $client The Elasticsearch client.
     * @param array $params The search parameters.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param ContainerInterface $container The dependency injection container.
     * @param array $config The configuration array.
     *
     * @return array The search results.
     */
    public function withOwnQuery(
        InterfaceFrontendService $frontendService,
        client $client,
        array $params,
        SalesChannelContext $saleschannelContext,
        ContainerInterface $container,
        array $config
    ) {
        $params['from'] = 0;
        $params['size'] = $config['ownFilterquery'];
        return $frontendService->search($client, $params, $saleschannelContext, $container);
    }

    /**
     * Sorts properties for the Ajax popup based on the provided configuration and products.
     *
     * @param array $products The array of products to process.
     * @param array $config The configuration array.
     * @param InterfaceFrontendService $frontendService The frontend service to use for searching.
     * @param Client $client The Elasticsearch client.
     * @param array $params The search parameters.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @param ContainerInterface $container The dependency injection container.
     *
     * @return array The sorted properties.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function sortAjaxPopUpProperties(
        array $products,
        array $config,
        InterfaceFrontendService $frontendService,
        client $client,
        array $params,
        SalesChannelContext $saleschannelContext,
        ContainerInterface $container
    ): array {
        $return = [];
        $optionMerker = [];

        // ---- Check if the 'properties' key exists in the configuration
        if (array_key_exists('properties', $config)) {
            // ---- Check if the 'properties' value is equal to '4'
            if ($config['properties'] === '4') {
                // ---- Check if the 'ownFilterquery' key exists in the configuration
                if (array_key_exists('ownFilterquery', $config)) {
                    // ---- Check if the 'ownFilterquery' value is greater than 0
                    if ($config['ownFilterquery'] > 0) {
                        $products = $this->withOwnQuery(
                            $frontendService,
                            $client,
                            $params,
                            $saleschannelContext,
                            $container,
                            $config
                        );
                    }
                }
                $hits = $products['hits']['hits'];

                // ---- Iterate through the search hits
                foreach ($hits as $hit) {
                    // ---- Iterate through the properties of each hit
                    foreach ($hit['_source']['properties'] as $property) {
                        // ---- Check if the option ID is not already in the marker array
                        if (!in_array($property["option_id"], $optionMerker)) {
                            $return[$property['property_id']][] = $property;
                            $optionMerker[] = $property["option_id"];
                        }
                    }

                    // ---- Check if the hit has children
                    if (array_key_exists('children', $hit['_source'])) {
                        // ---- Check if the children array is not empty
                        if (count($hit['_source']['children']) > 0) {
                            // ---- Iterate through the children
                            foreach ($hit['_source']['children'] as $childProperty) {
                                // ---- Check if the child has properties
                                if (array_key_exists("properties", $childProperty)) {
                                    // ---- Iterate through the properties of the child
                                    foreach ($childProperty["properties"] as $childPropertyItem) {
                                        // ---- Check if the child property option ID is not already in the marker array
                                        if (!in_array($childPropertyItem["option_id"], $optionMerker)) {
                                            $return[$childPropertyItem['property_id']][] = $childPropertyItem;
                                            $optionMerker[] = $childPropertyItem["option_id"];
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        return $return;
    }

    /**
     * Sets the sorted properties on the given fields and entity.
     *
     * @param array{channel : SalesChannelProductEntity} $fields The fields array, containing the sales channel product entity.
     * @param object $entity The entity to set the sorted properties on.
     * @param array $parameters The parameters array, containing the property group sorter.
     */
    public function setSortedProperties(array &$fields, &$entity, array $parameters): void
    {
        // ---- Check if the 'propertyGroupSorter' key exists in the parameters
        if (array_key_exists('propertyGroupSorter', $parameters)) {
            $properties = $entity->getProperties();
            $sortedProperties = $parameters['propertyGroupSorter']->sort($properties);
            $entity->setSortedProperties($sortedProperties);

            // ---- Check if the 'channel' field has the 'setSortedProperties' method
            if ((method_exists($fields['channel'], 'setSortedProperties'))) {
                $fields['channel']->setSortedProperties($sortedProperties);
                $fields['channel']->setProperties($properties);
            }
        }
    }
}