<?php

namespace Sisi\Search\Service;

use Symfony\Component\Console\Output\OutputInterface;

/**
 * Provides a service to display a formatted progress bar in the console.
 *
 * This class encapsulates the logic for generating and displaying a progress bar
 * that updates in the console, showing the completion percentage.
 */
class ProgressService
{

    /**
     * Displays a progress bar in the console.
     *
     * The progress bar visually represents the completion percentage.
     *
     * @param int $percentage The percentage of completion (0-100).
     * @param int $numDecimalPlaces The number of decimal places to display in the percentage.
     * @param OutputInterface | null $output The output interface to write to. If null, no output is produced.
     */
    public function showProgressBar($percentage, $numDecimalPlaces, $output): void
    {
        if ($output !== null) {
            // ---- Calculate the length of the percentage string
            $percentageStringLength = 4;
            if ($numDecimalPlaces > 0) {
                $percentageStringLength += ($numDecimalPlaces + 1);
            }

            // ---- Format the percentage string
            $percentageString = number_format($percentage, $numDecimalPlaces) . '%';
            $percentageString = str_pad($percentageString, $percentageStringLength, " ", STR_PAD_LEFT);
            $percentageStringLength += 3; // add 2 for () and a space before bar starts.

            // ---- Determine the terminal width
            $terminalWidth = `tput cols`;
            $terminalWidth = (int)$terminalWidth;

            // ---- Calculate the width of the progress bar
            $barWidth = $terminalWidth - ($percentageStringLength) - 2; // subtract 2 for [] around bar

            // ---- Calculate the number of filled and empty bars
            $numBars = (int)round(($percentage) / 100 * ($barWidth));
            $numEmptyBars = $barWidth - $numBars;

            // ---- Create the bar string
            $barsString = '[' . str_repeat("=", ($numBars)) . str_repeat(" ", ($numEmptyBars)) . ']';

            // ---- Create the output string
            $outputstring = "($percentageString) " . $barsString . "\r";

            // ---- Write the output string to the console
            $output->writeln($outputstring);
        }
    }
}