<?php

namespace Sisi\Search\Service;

use Doctrine\DBAL\Exception;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\MultiFilter;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Bridge\Monolog\Logger;

/**
 * This class is responsible for inserting data into the search index.
 * It handles various aspects such as translations, categories, manufacturers, properties, and prices.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class InsertMainMethode
{
    /**
     * Inserts product data into the search index.
     *
     * This method orchestrates the insertion of product data, including translations,
     * categories, manufacturers, properties, and prices, into the search index.
     *
     * @param EntitySearchResult $mappingValues The mapping values for the fields.
     * @param SalesChannelProductEntity $entity The product entity to insert.
     * @param string $lanugageId The language ID for the product.
     * @param Logger $logger The logging service for error reporting.
     * @param array $parameters An array of configuration parameters.
     * @param array $categoriesValue An array of category values.
     * @param array $categorieMerker An array to track processed categories.
     * @param array $propertiesMerker An array to track processed properties.
     * @param array $fields The array of fields to be indexed.
     * @throws Exception
     *
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    public function insert(
        EntitySearchResult        $mappingValues,
        SalesChannelProductEntity $entity,
        string                    $lanugageId,
        Logger                    $logger,
        array                     $parameters,
        array                     $categoriesValue,
        array                     &$categorieMerker,
        array                     &$propertiesMerker,
        array                     &$fields
    ): void
    {
        // ---- Initialize helper services
        $translationService = new TranslationService();
        $extendInsertService = new ExtendInsertService();
        $categorieService = new CategorieService();
        $searchkeyService = new SearchkeyService();
        $channelDataService = new ChannelDataService();
        $indexService = new IndexService();
        $priceService = new PriceService();
        $propertiesService = new PropertiesService();
        $insertService = new InsertService();

        $parentId = trim($entity->getParentId());

        // ---- Get product translations
        $translation = $translationService->getTranslationfields($entity->getTranslations(), $lanugageId, $parameters['config']);

        // ---- Check function and set data type
        $insertService->checkFunction(
            $mappingValues,
            $fields,
            $entity,
            $translation,
            $logger,
            $parameters['config'],
            'product',
            false,
            $parentId,
            []
        );
        $insertService->setDatetype($mappingValues, $fields, $entity, $translation);

        // ---- Set custom fields
        if ($translation) {
            $insertService->setCustomsFileds(
                $translation->getCustomFields(),
                $mappingValues,
                $fields,
                $logger,
                $parameters['config'],
                $parentId
            );
        }

        // ---- Insert search keywords
        $searchkeyService->insertSearchKey(
            $fields,
            $entity,
            $lanugageId,
            $mappingValues,
            $parameters['config'],
            $logger,
            $parentId,
            $insertService,
            $parameters['connection']
        );

        // ---- Handle categories
        if ($categorieService->strIndexCategorie($parameters['config'])) {
            $categoieStream = $categorieService->getProductStreamsCategories($entity);
            $categories = $entity->getCategories();
            $categorieService->getMergeCategories($categories, $categoieStream);
            foreach ($categories as $categorie) {
                $params['categorie'] = $categorie;
                $params['lanugageId'] = $lanugageId;
                $params['categoriesValue'] = $categoriesValue;
                $params['parentid'] = $parentId;
                $params['config'] = $parameters['config'];
                $params['categories'] = $categories;
                $extendInsertService->insertCategorie(
                    $translationService,
                    $params,
                    $mappingValues,
                    $fields,
                    $logger,
                    $insertService,
                    $categorieMerker
                );
            }
        }

        // ---- Handle manufacturers
        $manufacturers = $entity->getManufacturer();
        if ($indexService->checkManufacturer($manufacturers, $fields)) {
            $extendInsertService->setManufacturerValue(
                $insertService,
                $manufacturers,
                $parameters['config'],
                $translationService,
                $mappingValues,
                $fields,
                $logger,
                $lanugageId,
                $parentId
            );
        }

        $fields['id'] = $entity->getId();
        $fields['language'] = $lanugageId;
        $fields['channel'] = $channelDataService->getDatas($entity, $parameters['config'], $lanugageId, $parameters['urlGenerator']);
        $fields['properties'] = [];

        // ---- Handle properties
        $propertiesService->setSortedProperties($fields, $entity, $parameters);
        if ($entity->getSortedProperties()) {
            foreach ($entity->getSortedProperties() as $property) {
                $paramsPro['property'] = $property;
                $paramsPro['lanugageId'] = $lanugageId;
                $paramsPro['parentid'] = $parentId;
                $paramsPro['config'] = $parameters['config'];
                $extendInsertService->insertProperties(
                    $translationService,
                    $paramsPro,
                    $fields,
                    $mappingValues,
                    $logger,
                    $insertService,
                    $propertiesMerker
                );
            }
        }

        // ---- Add suggester field and insert price
        $extendInsertService->addSuggesterField($parameters['config'], $fields);
        $priceService->insertPrice($entity, $fields);
    }

    /**
     * Retrieves dynamic products based on the provided criteria and IDs.
     *
     * @param Criteria $criteria The criteria to apply to the product search.
     * @param array $merkerIdsFordynamicProducts An array of product IDs to filter by.
     * @param ContainerInterface $container The Symfony container to retrieve the product service.
     * @param SalesChannelContext $saleschannelContext The sales channel context.
     * @return EntitySearchResult
     */
    public function getDynamicproduct(
        Criteria            &$criteria,
        array               $merkerIdsFordynamicProducts,
        ContainerInterface  $container,
        SalesChannelContext $saleschannelContext
    )
    {
        $filter = [];
        foreach ($merkerIdsFordynamicProducts as $product) {
            $filter[] = new EqualsFilter('id', $product);
        }
        $criteria->addFilter(new MultiFilter(MultiFilter::CONNECTION_OR, $filter));
        $productService = $container->get('sales_channel.product.repository');
        return $productService->search($criteria, $saleschannelContext);
    }
}