<?php

namespace Sisi\Search\Service;

use DateTimeImmutable;
use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Category\CategoryEntity;
use Shopware\Core\Content\Product\Aggregate\ProductManufacturer\ProductManufacturerEntity;
use Shopware\Core\Content\Category\Aggregate\CategoryTranslation\CategoryTranslationEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Sisi\Search\Core\Content\Fields\Bundle\DBFieldsEntity;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\Console\Output\OutputInterface;
use Shopware\Core\Content\Category\CategoryCollection;

/**
 * This class provides methods to extend the insertion process of product data into the search index.
 * It handles tasks such as adding suggester fields, checking stock levels, removing duplicate entries,
 * inserting category and property information, and managing translations.
 *
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class ExtendInsertService
{
    /**
     * Adds a suggester field to the product data if the 'suggest' configuration is enabled.
     *
     * @param array $config The configuration array.
     * @param array $fields The product data fields array.
     */
    public function addSuggesterField(array $config, array &$fields): void
    {
        if (array_key_exists('suggest', $config) && array_key_exists('product_name', $fields)) {
            if ($config['suggest'] === '1') {
                $fields['product_name_trigram'] = $fields['product_name'];
            }
        }
    }

    /**
     * Displays a message indicating that a row is being inserted, based on the 'showinsert' configuration.
     *
     * @param array $parameters The parameters array.
     * @param OutputInterface $output The output interface for displaying the message.
     */
    public function displayInsertRow(array $parameters, OutputInterface $output): void
    {
        if (array_key_exists('showinsert', $parameters)) {
            if ($parameters['showinsert'] === '1') {
                $output->writeln('insert Row');
            }
        }
    }

    /**
     * Checks the stock level of a product and its variants, and determines whether the product should be indexed.
     *
     * @param SalesChannelProductEntity $entity The product entity.
     * @param array $parameters The parameters array, including configuration and database connection.
     * @param OutputInterface|null $output The output interface for displaying messages.
     * @param Logger $loggingService The logging service for recording messages.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @return bool True if the product should be indexed, false otherwise.
     */
    public function checkStockFromAllVaraints(
        $entity,
        $parameters,
        $output,
        $loggingService
    )
    {
        if (array_key_exists('config', $parameters)) {
            if (array_key_exists('onlymain', $parameters['config']) && array_key_exists('connection', $parameters)) {
                if ($parameters['config']['onlymain'] === 'stock') {
                    // ---- Fetch stock information for all variants of the product
                    $allVaraints = $this->dbqueryfromStockFromAllVaraints($parameters['connection'], $entity->getId());
                    $stock = 0;
                    $strWithOutMain = true;
                    foreach ($allVaraints as $varaint) {
                        $stock = $stock + $varaint['stock'];
                    }
                    if (array_key_exists('stockwithoutmain', $parameters['config'])) {
                        if ($parameters['config']['stockwithoutmain'] == 'yes') {
                            $strWithOutMain = false;
                        }
                    }
                    if ($strWithOutMain) {
                        $stock = $stock + $entity->getStock();
                    }
                    $categorien = $entity->getCategories();
                    $count = count($categorien->getElements());
                    if ($count === 0) {
                        return false;
                    }
                    $messasge = 'Product with the number ' . $entity->getProductNumber() . ' are  not indexed because the stock is empty';
                    if ($stock == 0) {
                        if ($output !== null) {
                            $output->writeln($messasge);
                        }
                        $loggingService->log('100', $messasge);
                        return false;
                    }
                }
            }
        }
        return true;
    }

    /**
     * Checks if a product should be removed due to being a duplicate, based on the 'nodouple' configuration.
     *
     * @param array $config The configuration array.
     * @param array $merker An array to keep track of processed product names.
     * @param SalesChannelProductEntity $entity The product entity.
     * @return bool True if the product should be kept, false if it's a duplicate.
     */
    public function checkRemoveDouble($config, &$merker, $entity)
    {
        if (array_key_exists('onlymain', $config)) {
            if ($config['onlymain'] === 'nodouple') {
                $name = $entity->getName();
                $name = str_replace(' ', '', $name);
                if (in_array($name, $merker)) {
                    return false;
                } else {
                    $merker[] = $name;
                }
            }
        }
        return true;
    }

    /**
     * Executes a database query to retrieve stock information for all variants of a product.
     *
     * @param Connection $connection The database connection.
     * @param string $id The ID of the parent product.
     * @return array An array of stock information for the variants.
     */
    public function dbqueryfromStockFromAllVaraints(connection $connection, string $id): array
    {
        $query = $connection->createQueryBuilder()
            ->select(['HEX(id),stock,product_number,manufacturer_number, HEX(parent_id)'])
            ->from('product')
            ->where('parent_id = UNHEX(:id)')
            ->setParameter(':id', $id);
        return $query->execute()->fetchAll();
    }


    /**
     * Displays the last line of the progress bar, based on the 'backend' configuration.
     *
     * @param array $parameters The parameters array.
     * @param ProgressService $progress The progress service.
     * @param OutputInterface|null $output The output interface for displaying the progress bar.
     */
    public function echoLastLine($parameters, $progress, $output): void
    {
        if (!array_key_exists('backend', $parameters)) {
            $progress->showProgressBar(100, 2, $output);
        }
    }

    /**
     * Sets the manufacturer value for a product, including handling translations and custom fields.
     *
     * @param InsertService $insertService The insert service.
     * @param ProductManufacturerEntity|null $manufacturers The manufacturer entity.
     * @param array $config The configuration array.
     * @param TranslationService $haendlerTranslation The translation service.
     * @param EntitySearchResult $mappingValues The mapping values.
     * @param array $fields The product data fields array.
     * @param Logger $logger The logging service.
     * @param string $lanugageId The language ID.
     * @param string|null $parentId The parent ID.
     */
    public function setManufacturerValue(
        InsertService $insertService,
                      $manufacturers,
                      $config,
                      $haendlerTranslation,
                      $mappingValues,
                      &$fields,
                      $logger,
                      $lanugageId,
                      $parentId
    )
    {
        if (method_exists($manufacturers, 'getTranslations')) {
            $translation = $haendlerTranslation->getTranslationfields($manufacturers->getTranslations(), $lanugageId, $config);
            $insertService->checkFunction(
                $mappingValues,
                $fields,
                $manufacturers,
                $translation,
                $logger,
                $config,
                $table = 'manufacturer',
                $isArrayFrom = true
            );
            if ($translation) {
                $custoMmanufacturer = $translation->getCustomFields();
                $insertService->setCustomsFileds(
                    $custoMmanufacturer,
                    $mappingValues,
                    $fields,
                    $logger,
                    $config,
                    $parentId
                );
            } else {
                $custoMmanufacturer = $manufacturers->getCustomFields();
                $insertService->setCustomsFileds(
                    $custoMmanufacturer,
                    $mappingValues,
                    $fields,
                    $logger,
                    $config,
                    $parentId
                );
            }
        } else {
            if (method_exists($manufacturers, 'getCustomFields')) {
                $custoMmanufacturer = $manufacturers->getCustomFields();
                $insertService->setCustomsFileds(
                    $custoMmanufacturer,
                    $mappingValues,
                    $fields,
                    $logger,
                    $config,
                    $parentId
                );
            }
            $insertService->checkFunction(
                $mappingValues,
                $fields,
                $manufacturers,
                $manufacturers,
                $logger,
                $config,
                $table = 'manufacturer',
                $isArrayFrom = true,
                $parentId
            );
        }
    }

    /**
     * Inserts category information into the product data, including handling translations and breadcrumbs.
     *
     * @param TranslationService $translationService The translation service.
     * @param array $params An array of parameters including category, language ID, and configuration.
     * @param EntitySearchResult $mappingValues The mapping values.
     * @param array $fields The product data fields array.
     * @param Logger $logger The logging service.
     * @param InsertService $insertService The insert service.
     * @param array $categorieMerker An array to keep track of processed category IDs.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function insertCategorie(
        TranslationService $translationService,
        array              $params,
        EntitySearchResult $mappingValues,
        array              &$fields,
        Logger             $logger,
        InsertService      $insertService,
        array              &$categorieMerker
    ): void
    {
        $categoryFields = $translationService->getTranslationfields(
            $params['categorie']->getTranslations(),
            $params['lanugageId'],
            $params['config']
        );
        $categorieId = trim($params['categorie']->getId());
        $strCategory = in_array($categorieId, $params['categoriesValue']);
        $merkerCatId = [];
        $heanderinsert = new CategorieInsertService();
        if ($strCategory) {
            $index = 0;
            $help = [];
            foreach ($params['categories']->getElements() as $category) {
                if ($category->getType() === 'page') {
                    $categorieTranslation = $translationService->getTranslationfields(
                        $category->getTranslations(),
                        $params['lanugageId'],
                        $params['config']
                    );
                    $categorieid = trim($category->getId());
                    if (!in_array($categorieid, $merkerCatId)) {
                        $help[$index]['category_id'] = $categorieid;
                        $merkerCatId[] = $categorieid;
                        /** @var DBFieldsEntity $mappingValue */
                        foreach ($mappingValues as $mappingValue) {
                            if ($mappingValue->getTablename() === 'category') {
                                $methode = "get" . ucfirst($mappingValue->getName());
                                $prefix = strtolower($mappingValue->getPrefix());
                                $indexName = 'category_' . $prefix . strtolower($mappingValue->getName());
                                $this->_insertCatgorieToArray(
                                    $categorieTranslation,
                                    $index,
                                    $category,
                                    $methode,
                                    $indexName,
                                    $help
                                );
                                $help[$index]['category_breadcrumb'] = $this->mergebreadcrumb($categorieTranslation, $category, $params['config']);
                            }
                        }
                    }
                }
                $index++;
            }
            $fields['categories'] = $help;
            $fields['category_id'] = $categorieId;
            $insertService->checkFunction(
                $mappingValues,
                $fields,
                $params['categorie'],
                $categoryFields,
                $logger,
                $params['config'],
                'category',
                true,
                $params['parentid'],
                ['multivalue' => $params['categories']]
            );
            if ($categoryFields) {
                $insertService->setCustomsFileds(
                    $categoryFields->getCustomFields(),
                    $mappingValues,
                    $fields,
                    $logger,
                    $params['config'],
                    $params['parentid']
                );
            }
            $categorieMerker[] = $categorieId;
            $heanderinsert->mergeCategorieBreadcrumbInTheproduct($fields, $params['config']);
        }
    }

    /**
     * Inserts category data into an array.
     *
     * @param CategoryEntity|bool $categorieTranslation The category translation entity.
     * @param int $index The index of the category.
     * @param CategoryEntity $categorie The category entity.
     * @param string $name The name of the method to call.
     * @param string $indexName The name of the index in the array.
     * @param array $help The array to insert the data into.
     */
    private function _insertCatgorieToArray(
        $categorieTranslation,
        $index,
        $categorie,
        $name,
        string $indexName,
        &$help
    )
    {

        if ($categorieTranslation !== false && method_exists($categorieTranslation, $name)) {
            $value = trim($categorieTranslation->$name());
        }
        if (method_exists($categorie, $name) && empty($value)) {
            $value = trim($categorie->$name());
        }

        if (!empty($value)) {
            $help[$index][$indexName] = $value;
        }
    }

    /**
     * Merges the breadcrumb of a category into a string.
     *
     * @param CategoryTranslationEntity $categorieTranslation The category translation entity.
     * @param CategoryEntity|bool $categorie The category entity.
     * @param array $config The configuration array.
     * @return string The merged breadcrumb string.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function mergebreadcrumb($categorieTranslation, $categorie, $config)
    {
        $depth = 0;
        $seperator = " ";
        $values = null;

        if (array_key_exists('breadcrumbDepth', $config)) {
            $depth = $config['breadcrumbDepth'];
        }

        if ($categorieTranslation == null or $categorieTranslation == false) {
            return '';
        }

        if (array_key_exists('breadcrumbseparator', $config)) {
            $seperator = $config['breadcrumbseparator'];
        }
        if ($categorieTranslation !== false) {
            $values = $categorieTranslation->getBreadcrumb();
        }

        $value = "";
        $orginal = $categorie->getPlainBreadcrumb();
        if ($values !== null) {
            $values = $this->sortbreadcrumb($orginal, $values);
            $index = 0;
            foreach ($values as $item) {
                if (!empty($item) && ($index >= $depth)) {
                    $value .= $seperator . $item;
                }
                $index++;
            }
        }
        return trim($value);
    }

    /**
     * Sorts the breadcrumb array.
     *
     * @param array $orgianl The original breadcrumb array.
     * @param array $translated The translated breadcrumb array.
     * @return array The sorted breadcrumb array.
     */
    private function sortbreadcrumb(array $orgianl, array $translated): array
    {
        $return = [];
        foreach ($orgianl as $key => $item) {
            $return[$key] = $translated[$key];
        }
        return $return;
    }

    /**
     * Inserts property information into the product data, including handling translations.
     *
     * @param TranslationService $translationService The translation service.
     * @param array $params An array of parameters including property, language ID, and configuration.
     * @param array $fields The product data fields array.
     * @param EntitySearchResult $mappingValues The mapping values.
     * @param Logger $logger The logging service.
     * @param InsertService $insertService The insert service.
     * @param array $propertiesMerker An array to keep track of processed property IDs.
     */
    public function insertProperties(
        TranslationService $translationService,
        array              $params,
        array              &$fields,
        EntitySearchResult $mappingValues,
        Logger             $logger,
        InsertService      $insertService,
        array              &$propertiesMerker
    ): void
    {
        $propertyId = trim($params['property']->getId());
        $propertyname = $params['property']->getName();
        $translationGroup = $params['property']->getTranslations();
        if ($translationGroup !== null) {
            $translation = $translationService->getTranslationfields(
                $translationGroup,
                $params['lanugageId'],
                $params['config'],
            );
            if ($translation !== false && $translation !== null) {
                $propertyname = $translation->getName();
            }
        }
        foreach ($params['property']->getOptions() as $option) {
            $optionTranslation = $translationService->getTranslationfields(
                $option->getTranslations(),
                $params['lanugageId'],
                $params['config'],
            );
            $name = $option->getName();
            if ($optionTranslation !== false) {
                $name = $optionTranslation->getName();
            }
            $fields['properties'][] = [
                'property_id'    => $propertyId,
                'property_group' => $propertyname,
                'option_name'    => $name,
                'option_id'      => $option->getId()
            ];
            $insertService->checkFunction(
                $mappingValues,
                $fields,
                $params['property'],
                $optionTranslation,
                $logger,
                $params['config'],
                'properties',
                true,
                $params['parentid']
            );
            $propertiesMerker[] = $propertyId;
        }
    }

    /**
     * Gets the translated value of a field, handling custom fields and default language fallback.
     *
     * @param mixed $translation The translation entity.
     * @param string $name The name of the field to retrieve.
     * @param mixed $entity The entity to retrieve the field from.
     * @param array $ext An array of extra parameters.
     * @param array $config The configuration array.
     * @return string The translated value of the field.
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    public function getTranslation($translation, $name, $entity, $ext, $config = [])
    {
        $value = '';
        if (!array_key_exists("multivalue", $ext)) {
            if (is_string($translation) || is_object($translation)) {
                if (method_exists($translation, $name)) {
                    $value = $translation->$name();
                }
            }
            if (empty($value) && ($translation !== false)) {
                $customsFields = $translation->getCustomFields();
                if ($customsFields !== false) {
                    if (is_array($customsFields)) {
                        $index = substr($name, 3);
                        if (array_key_exists($index, $customsFields)) {
                            $value = $customsFields[$index];
                        }
                    }
                }
            }
            if (empty($value)) {
                $strlanguage = true;
                if (array_key_exists('configLanguage', $config)) {
                    if (!empty($config['configLanguage'])) {
                        $strlanguage = false;
                    }
                }
                if (method_exists($entity, $name)) {
                    $value = $entity->$name();
                }
                if ($strlanguage) {
                    $value = $this->getValueFromDefaultLanguage($name, $entity, $value);
                } else {
                    $value = $this->getValueFromConfigLanguage($name, $entity, $value, $config['configLanguage']);
                }
            }
        } else {
            $index = 0;
            foreach ($ext['multivalue'] as $entry) {
                if ($index === 0) {
                    $value = $entry->$name();
                } else {
                    $entryvalue = $entry->$name();
                    if (gettype($entryvalue) === 'string') {
                        $value .= " " . $entryvalue;
                    }
                }
                $index++;
            }
        }
        return $value;
    }


    /**
     * Gets the value from the default language if the value is empty.
     *
     * @param string $name The name of the field.
     * @param mixed $entity The entity.
     * @param string $value The current value.
     * @return string The value from the default language or the original value.
     */
    public function getValueFromDefaultLanguage($name, $entity, $value)
    {
        if (empty($value)) {
            if ($entity->getTranslations() && method_exists($entity->getTranslations(), 'getElements')) {
                $elements = $entity->getTranslations()->getElements();
                foreach ($elements as $transElement) {
                    if ($transElement->getLanguageId() === Defaults::LANGUAGE_SYSTEM) {
                        if (method_exists($transElement, $name)) {
                            return $transElement->$name();
                        }
                    }
                }
            }
        }
        return $value;
    }

    /**
     * Gets the value from the configured language if the value is empty.
     *
     * @param string $name The name of the field.
     * @param mixed $entity The entity.
     * @param string $value The current value.
     * @param string $languageid The language ID.
     * @return string The value from the configured language or the original value.
     */
    public function getValueFromConfigLanguage($name, $entity, $value, $languageid)
    {

        if (empty($value)) {
            if ($entity->getTranslations() && method_exists($entity->getTranslations(), 'getElements')) {
                $elements = $entity->getTranslations()->getElements();
                foreach ($elements as $transElement) {
                    if ($transElement->getLanguageId() === $languageid) {
                        if (method_exists($transElement, $name)) {
                            return $transElement->$name();
                        }
                    }
                }
            }
        }
        return $value;
    }
}