<?php

namespace Sisi\Search\Service;

/**
 * This class provides extended search functionalities, including stripping URLs,
 * setting AND operators for search queries, handling highlight fields, and
 * determining query fields based on configuration.
 * @SuppressWarnings(PHPMD)
 */
class ExtSearchService
{
    /**
     * Strips specified characters from a given term based on the provided configuration.
     *
     * @param string|null $term The term to strip characters from.
     * @param array $config The configuration array containing the characters to strip.
     * @return string|null The stripped term, or an empty string if the term is null after stripping.
     */
    public function stripUrl($term, $config)
    {
        // ---- Check if the 'strip' key exists in the configuration array
        if (array_key_exists('strip', $config)) {
            // ---- Check if the 'strip' value is not empty
            if (!empty($config['strip'])) {
                // ---- Explode the string of characters to strip into an array
                $charaters = explode("\n", $config['strip']);
                // ---- Iterate over each character and remove it from the term
                foreach ($charaters as $charater) {
                    $term = str_replace($charater, "", $term);
                }
            }
        }
        // ---- Handle null term
        if ($term === null) {
            $term = "";
        }
        return $term;
    }

    /**
     * Sets the AND operator for search terms based on provided parameters, fields, and system configuration.
     *
     * @param array $terms An array of search terms.
     * @param array $params An array of search parameters.
     * @param array $fields An array of search fields.
     * @param array $systemConfig An array of system configuration settings.
     * @param string $match The matching type.
     * @param bool $str A reference to a boolean indicating whether the AND operator is set.
     * @return array The modified search parameters array.
     */
    public function setAndOperator(array $terms, array $params, array $fields, array $systemConfig, string $match, bool &$str): array
    {
        // ---- Check if any of the terms 'ma', 'cat', or 'pro' are not null
        if ($terms['ma'] != null || $terms['cat'] != null || $terms['pro'] != null) {
            $str = true;
            $return = [
                'index' => $params['index'],
                'from' => $params['from'],
                'size' => $params['size']
            ];
            $newfields = [];

            // ---- Iterate over the fields to filter out specific fields based on the terms
            foreach ($fields as $index => $field) {
                if (array_key_exists($match, $field)) {
                    foreach ($field[$match] as $key => $item) {
                        if ($terms['pro'] !== null && $key === 'properties.option_name' || $terms['ma'] != null && $key === 'manufacturer_name') {
                        } else {
                            $newfields[] = $field;
                        }
                    }
                }
            }

            // ---- Build the query for product terms
            if (!empty($terms['product'])) {
                $query["bool"]['should'] = $newfields;
                $return['body']['query']['bool']['must'][] = $query;
            }

            // ---- Build the query for manufacturer terms
            if (!empty($terms['ma'])) {
                $Manfacture['bool']['should'][0]['match']['manufacturer_id']['query'] = $terms['ma'];
                $Manfacture['bool']['should'][1]['match']['manufacturer_name']['query'] = $terms['ma'];
                $return['body']['query']['bool']['must'][] = $Manfacture;
            }

            // ---- Build the query for category terms
            if (!empty($terms['cat'])) {
                $cat["path"] = "categories";
                $cat["query"]["bool"]['should'][0]['match']['categories.category_name'] = trim($terms['cat']);
                $cat["query"]["bool"]['should'][1]['match']['categories.category_id'] = trim($terms['cat']);
                $return['body']['query']['bool']['must'][]["nested"] = $cat;
            }

            // ---- Build the query for properties terms
            if ($terms['pro'] !== null) {
                foreach ($terms['pro'] as $pro) {
                    $properties = [];
                    if (array_key_exists('properties', $systemConfig)) {
                        $properties["path"] = "properties";
                        $properties["query"]["bool"]['should'][0]['match']['properties.option_id'] = trim($pro);
                        $properties["query"]["bool"]['should'][1]['match']['properties.option_name'] = trim($pro);
                    }
                    $heandlervarianten = new VariantenService();
                    if ($heandlervarianten->conditionFunction($systemConfig)) {
                        $propertiesChild["nested"] = $this->mergeFilterQueryChildren($pro);
                        $return['body']['query']['bool']['must'][]["bool"]['should'] = [
                            0 => ["nested" => $properties],
                            1 => $propertiesChild
                        ];
                    } else {
                        $return['body']['query']['bool']['must'][]["nested"] = $properties;
                    }
                }
            }

            // ---- Add highlight to the return array
            if (array_key_exists('highlight', $params['body'])) {
                $return['body']['highlight'] = $params['body']['highlight'];
            }
            return $return;
        }
        return $params;
    }

    /**
     * Merges filter query for children properties.
     *
     * @param string $pro The property value.
     * @return array The merged filter query for children properties.
     */
    private function mergeFilterQueryChildren(string $pro)
    {
        $properties["path"] = "children.properties";
        $properties["query"]["bool"]['should'][0]['match']['children.properties.option_id'] = trim($pro);
        $properties["query"]["bool"]['should'][1]['match']['children.properties.option_name'] = trim($pro);
        $propertiesArray["path"] = "children";
        $propertiesArray["query"]['nested'] = $properties;
        return $propertiesArray;
    }

    /**
     * Retrieves highlight fields from an array of fields.
     *
     * @param array $fields An array of fields.
     * @return array An array of highlight fields.
     */
    public function getHighlightFields(array $fields): array
    {
        $return = [];
        // ---- Iterate over each field item
        foreach ($fields as $fieldsItem) {
            // ---- Check if the 'match' key exists in the field item
            if (array_key_exists('match', $fieldsItem)) {
                $key = array_key_first($fieldsItem['match']);
                $return[] = [$key => new \stdClass()];
            }
            // ---- Check if the 'match_phrase_prefix' key exists in the field item
            if (array_key_exists('match_phrase_prefix', $fieldsItem)) {
                $key = array_key_first($fieldsItem['match_phrase_prefix']);
                $return[] = [$key => new \stdClass()];
            }
            // ---- Check if the 'match_phrase' key exists in the field item
            if (array_key_exists('match_phrase', $fieldsItem)) {
                $key = array_key_first($fieldsItem['match_phrase']);
                $return[] = [$key => new \stdClass()];
            }
        }
        return $return;
    }

    /**
     * Determines whether to query fields based on the table name and configuration.
     *
     * @param string $tablename The name of the table.
     * @param array $config The configuration array.
     * @return bool True if the fields should be queried, false otherwise.
     */
    public function strQueryFields(string $tablename, array $config): bool
    {
        $str = true;
        // ---- Check if the table name is 'category' and the 'categorien' configuration is set to '2'
        if ($tablename === 'category' && $config['categorien'] == '2') {
            $str = false;
        }
        // ---- Check if the 'properties' key exists in the configuration array
        if (array_key_exists('properties', $config)) {
            // ---- Check if the table name is 'properties' and the 'properties' configuration is set to '2'
            if ($tablename === 'properties' && $config['properties'] == '2') {
                $str = false;
            }
        }
        // ---- Check if the table name is 'manufacturer' and the 'manufacturer' configuration is set to '2'
        if ($tablename === 'manufacturer' && $config['manufacturer'] == '2') {
            $str = false;
        }

        return $str;
    }
}