<?php

namespace Sisi\Search\Service;

use Doctrine\DBAL\Connection;
use Elasticsearch\Client;
use Exception;
use Symfony\Bridge\Monolog\Logger;

/**
 * This class provides functionality to delete inactive products from Elasticsearch
 * and retrieve a list of inactive products from the database.
 */
class ExInaktiveExService
{
    /**
     * Deletes products from the Elasticsearch server.
     *
     * @param array $products An array of product data, each containing an 'id'.
     * @param Client $client The Elasticsearch client.
     * @param string $indexName The name of the Elasticsearch index.
     * @param Logger $logger The logger instance.
     */
    public function delteInESServer(array $products, Client $client, string $indexName, Logger $logger)
    {
        foreach ($products as $product) {
            $params = [
                'index' => $indexName,
                'id' => strtolower($product['id'])
            ];
            try {
                $client->delete($params);
            } catch (Exception $e) {
                $logger->log(100, $e->getMessage());
            }
        }
    }

    /**
     * Retrieves a list of inactive products from the database based on the provided parameters.
     *
     * @param Connection $connection The database connection.
     * @param array $parameters An array of parameters including 'type', 'channelId', 'limit', and 'offset'.
     * @return array An array of product IDs.
     */
    public function getAllInaktiveProducts(connection $connection, array $parameters): array
    {
        $sql = "";

        // ---- Build the SQL query based on the 'type' parameter
        if ($parameters['type'] === 'inaktive') {
            $sql = "SELECT  HEX(product.id) AS `id`
                FROM product
                     LEFT JOIN product_visibility
                     ON product.id = product_visibility.product_id
                WHERE product.active = 0 and  product_visibility.sales_channel_id = UNHEX(:channelId)";
        }

        if ($parameters['type'] === 'clean') {
            $sql = "SELECT  HEX(product.id) AS `id`
                FROM product
                     LEFT JOIN product_visibility
                     ON product.id = product_visibility.product_id
                WHERE (product.stock < 0  OR product.available =  0 OR is_closeout = 1)
                and  product_visibility.sales_channel_id = UNHEX(:channelId)";
        }

        // ---- Set a default limit if not provided
        if (!array_key_exists('limit', $parameters)) {
            $parameters['limit'] = 1000;
        }

        // ---- Add limit and offset to the query if provided
        if (array_key_exists('offset', $parameters) && array_key_exists('limit', $parameters)) {
            $sql .= " LIMIT " . $parameters['offset'];
            $sql .= "," . $parameters['limit'];
        }

        return $connection->fetchAllAssociative($sql, ['channelId' => $parameters['channelId']]);
    }
}