<?php

namespace Sisi\Search\Service;

use Psr\Container\ContainerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;

/**
 * This class provides backend indexing services for the Sisi search module.
 * It handles starting, deleting, and checking the status of index processes.
 */
class BackendIndexService
{
    /**
     * @var integer
     */
    private $pid;


    /**
     * @var string
     */
    private $command;

    /**
     * BackendIndexService constructor.
     *
     * @param string $pfad The path to the console command.
     */
    public function __construct(string $pfad = '')
    {
        $this->command = $pfad . "/";
    }


    /**
     * Starts the indexing process.
     *
     * @param array $result An array containing the parameters for the indexing process.
     * @param array $config An array containing configuration settings.
     *
     * @return int The process ID (PID) of the started indexing process.
     */
    public function startIndex(array $result, array $config): int
    {
        $memory = '';
        $nohup = 'nohup ';
        $phpPad = $this->getPhpPfad($config);
        $command = $memory . $this->command .
            'bin/console sisi-Produkt-Stepindex:start shopID=' . escapeshellarg($result['shopID']);
        $output = [];

        $command .= $this->mergeIndexCommand($result, 'languageID');

        if (!empty($result['limit']) && $result['limit'] > 0) {
            $command .= ' limit=' . escapeshellarg($result['limit']);
        }

        $command .= $this->mergeIndexCommand($result, 'main');
        $command .= $this->updaterIndexer($result);
        $command .= $this->mergeIndexCommand($result, 'time');
        $command .= ' backend="1" ';
        $command .= ' > ' . $this->command . 'var/log/sisi.log 2>&1 & echo $!';

        if (!empty($result['memory']) && is_numeric($result['memory'])) {
            $memory = ' -d memory_limit=' . escapeshellarg($result['memory'] . 'M') . ' ';
        }

        if (!empty($phpPad)) {
            $memory = $phpPad . $memory;
        }

        $command = $nohup . $memory . $command;

        exec($command, $output);

        if (array_key_exists(0, $output)) {
            $this->pid = (int)$output[0];
        }
        return $this->pid;
    }

    /**
     * Determines if the indexer should be updated and merges the update command.
     *
     * @param array $result An array containing the parameters for the indexing process.
     *
     * @return string The update command if the 'update' key exists in the result array and its value is '1' or '2', otherwise an empty string.
     */
    private function updaterIndexer(array $result): string
    {
        if (array_key_exists('update', $result)) {
            if ($result['update'] === '1' || $result['update'] === '2') {
                return $this->mergeIndexCommand($result, 'update');
            }
        }
        return "";
    }

    /**
     * Retrieves the PHP path from the configuration.
     *
     * @param array $config An array containing configuration settings.
     *
     * @return string The PHP path, or 'php ' if not specified in the configuration.
     */
    private function getPhpPfad(array $config): string
    {
        $phpPad = 'php ';
        if (array_key_exists('phpPad', $config)) {
            if (!empty($config['phpPad'])) {
                $phpPad = $config['phpPad'] . ' ';
            }
        }
        return $phpPad;
    }

    /**
     * Merges an index command with a given key and value from the result array.
     *
     * @param array  $result An array containing the parameters for the indexing process.
     * @param string $key    The key to retrieve the value from the result array.
     *
     * @return string The merged index command, or an empty string if the key is not found or the value is empty.
     */
    private function mergeIndexCommand(array $result, string $key): string
    {
        $command = '';
        if (!empty($result[$key])) {
            $command .= ' ' . $key . '=' . escapeshellarg($result[$key]);
        }
        return $command;
    }

    /**
     * Deletes the index based on the provided parameters.
     *
     * @param array $result An array containing the parameters for the deletion process.
     * @param array $config An array containing configuration settings.
     *
     * @return int The process ID (PID) of the deletion process.
     */
    public function delete(array $result, array $config): int
    {
        $phpPad = '';
        if (array_key_exists('phpPad', $config)) {
            $phpPad = $config['phpPad'] . ' ';
        }
        $command = 'nohup ' . $phpPad . $this->command . 'bin/console sisi-Produkt-index:delete ';

        // ---- Add languageID to the command if it exists
        if (!empty($result['languageID'])) {
            $command .= ' languageID=' . escapeshellarg($result['languageID']);
        }

        // ---- Add all to the command if it exists
        if (!empty($result['all']) && $result['all'] > 0) {
            $command .= ' all=' . escapeshellarg($result['all']);
        }

        // ---- Add shopID to the command if it exists
        if (!empty($result['shopID'])) {
            $command .= ' shopID=' . escapeshellarg($result['shopID']);
        }

        $command .= ' > ' . $this->command . 'var/log/sisi.log 2>&1 & echo $!';

        exec($command, $output);

        if (array_key_exists(0, $output)) {
            $this->pid = (int)$output[0];
        }
        return $this->pid;
    }

    /**
     * Deletes inactive products from the index.
     *
     * @param array $result An array containing the parameters for the deletion process.
     * @param array $config An array containing configuration settings.
     *
     * @return int The process ID (PID) of the deletion process.
     */
    public function inaktive(array $result, array $config): int
    {
        $phpPad = '';
        if (array_key_exists('phpPad', $config)) {
            $phpPad = $config['phpPad'] . ' ';
        }


        $command = 'nohup ' . $phpPad . $this->command . 'bin/console  sisi-Produkt-inaktive:delete';

        // ---- Add languageID to the command if it exists
        if (!empty($result['languageID'])) {
            $command .= ' languageID=' . escapeshellarg($result['languageID']);
        }

        // ---- Add shopID to the command if it exists
        if (!empty($result['shopID'])) {
            $command .= ' shopID=' . escapeshellarg($result['shopID']);
        }

        $command .= ' > ' . $this->command . 'var/log/sisi.log 2>&1 & echo $!';

        exec($command, $output);

        if (array_key_exists(0, $output)) {
            $this->pid = (int)$output[0];
        }
        return $this->pid;
    }

    /**
     * Retrieves the status of a process.
     *
     * @param string|int $pid The process ID.
     *
     * @return string The output of the `ps` command.
     */
    public function status($pid)
    {
        $command = 'ps -p ' . $pid;
        return shell_exec($command);
    }

    /**
     * Retrieves the content of the Sisi log file.
     *
     * @return array An array containing each line of the log file.
     */
    public function getLog(): array
    {
        $pfad = $this->command . 'var/log/sisi.log';
        return file($pfad);
    }

    /**
     * Retrieves all sales channels.
     *
     * @param ContainerInterface $container The service container.
     *
     * @return array An array of sales channel elements.
     */
    public function getChannel(ContainerInterface $container): array
    {
        $saleschannel = $container->get('sales_channel.repository');
        $criteriaChannel = new Criteria();
        $contexhaendler = new ContextService();
        $context = $contexhaendler->getContext();
        return $saleschannel->search($criteriaChannel, $context)->getEntities()->getElements();
    }

    /**
     * Retrieves all languages.
     *
     * @param ContainerInterface $container The service container.
     *
     * @return array An array of language elements.
     */
    public function getLanguages(ContainerInterface $container): array
    {
        $language = $container->get('language.repository');
        $criteriaChannel = new Criteria();
        $contexhaendler = new ContextService();
        $context = $contexhaendler->getContext();
        return $language->search($criteriaChannel, $context)->getEntities()->getElements();
    }
}