<?php

namespace Sisi\Search\Decorater;

use Elasticsearch\Client;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Sisi\Search\Service\ContextService;
use Sisi\Search\Service\QuerylogService;
use Symfony\Component\Console\Output\OutputInterface;
use Sisi\Search\ServicesInterfaces\InterfaceQuerylogSearchService;

/**
 * This class is responsible for searching and inserting data into the query log.
 * It interacts with Elasticsearch to perform these operations.
 */
class QuerylogSearchService implements InterfaceQuerylogSearchService
{

    /**
     * Searches the query log based on the provided parameters.
     *
     * @param array $config Configuration array.
     * @param Client $client Elasticsearch client.
     * @param string $languageId Language ID.
     * @param array $terms Search terms.
     * @param string $saleschannelId Sales channel ID.
     *
     * @return array Search results from Elasticsearch.
     */
    public function searchQuerlog(
        array $config,
        Client $client,
        string $languageId,
        array $terms,
        $saleschannelId
    ): array {
        // ---- Initialize QuerylogService
        $heandlerQuerylog = new QuerylogService();
        $parameters['channelId'] = $saleschannelId;
        $parameters['lanuageName'] = $languageId;
        $indexname = $heandlerQuerylog->createIndexName($parameters);
        $size = 5;

        // ---- Check if querylogsize is defined in the config
        if (array_key_exists('querylogsize', $config)) {
            $size = $config['querylogsize'];
        }

        // ---- Check if querylog is enabled in the config
        if (array_key_exists('querylog', $config)) {
            if ($config['querylog'] === '1') {
                $must['bool']['must'][0]['match']['term'] = $terms['product'];
                $must['bool']['must'][1]['match']['language_id'] = $languageId;
                $params = [
                    'index' => $indexname,
                    'size' => $size,
                    'from'  => 0,
                    'body' => [
                        "query" => $must,
                        'aggs' => [
                            'historyTerms' => [
                                "terms" => [
                                    "field" => "term.keyword",
                                    "size" => $size
                                ],
                            ],
                            'historyProductname' => [
                                "terms" => [
                                    "field" => "product_name.keyword",
                                    "size" => $size
                                ]
                            ]
                        ],
                    ],

                ];
                return $client->search($params);
            }
        }
        return [];
    }

    /**
     * Inserts data into the Elasticsearch index.
     *
     * @param array $fields Data to be inserted.
     * @param string $esIndex Elasticsearch index name.
     * @param Client $client Elasticsearch client.
     *
     * @return array Result of the index operation.
     */
    public function insert(array $fields, string $esIndex, Client $client): array
    {
        // ---- Initialize ContextService
        $contexthaendler = new ContextService();
        $params = [
            'index' => $esIndex,
            'id' => $contexthaendler->getRandomHex(),
            'body' => $fields
        ];
        return $client->index($params);
    }
}