<?php

namespace Sisi\Search\Controller;

use Doctrine\DBAL\Driver\Exception;
use Sisi\Search\Service\V2\IndexService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * This controller handles requests related to Elasticsearch index management.
 * It provides actions for deleting, retrieving, and checking the status of Elasticsearch indices.
 * @Route(defaults={"_routeScope"={"api"}})
 */
class ElasticsearchIndexController extends AbstractController
{
    private IndexService $indexService;

    /**
     * ElasticsearchIndexController constructor.
     *
     * @param IndexService $indexService The index service.
     */
    public function __construct(
        IndexService $indexService
    ) {
        $this->indexService = $indexService;
    }


    /**
     * Deletes an Elasticsearch index based on the provided index name.
     *
     * @Route("api/_action/sisi/elasticsearch/deleteIndex", name="api.action.deleteIndex", methods={"POST"} )
     * @param Request $request The request object containing the index name.
     * @return JsonResponse A JSON response indicating the status of the deletion operation.
     * @throws Exception
     * @throws \Doctrine\DBAL\Exception
     */
    public function deleteIndex(Request $request): JsonResponse
    {
        $indexName = $request->get('indexName');
        if ($indexName) {
            $result = $this->indexService->deleteIndex($indexName);
        } else {
            $result['status'] = "No index selected";
        }
        return new JsonResponse($result);
    }

    /**
     * Retrieves a list of Elasticsearch indices for a given sales channel.
     *
     * @Route("api/_action/sisi/elasticsearch/getIndexes", name="api.action.getIndexes", methods={"POST"} )
     *
     * @param Request $request The request object containing the sales channel ID.
     * @return JsonResponse A JSON response containing the list of indices with their statistics.
     * @throws Exception
     * @throws \Doctrine\DBAL\Exception
     */
    public function getIndexes(Request $request): JsonResponse
    {
        $salesChannelId = $request->get('salesChannelId');
        $indices = $this->indexService->getIndexes($salesChannelId);
        $statistics = $this->indexService->getStatistics($salesChannelId);
        $list = [];

        // ---- Iterate through the indices and add statistics
        foreach ($indices as $index) {
            $row = $index;
            $indexStatistics = $statistics['indices'];

            // ---- Check if statistics exist for the current index
            if (array_key_exists($index['index'], $indexStatistics)) {
                $indexStat = $indexStatistics[$index['index']];
                $row['indexSize'] = $indexStat['total']['store']['size_in_bytes'];
                $row['docs'] = $indexStat['primaries']['docs']['count'];
            }
            $list[] = $row;
        }

        return new JsonResponse($list);
    }

    /**
     * Retrieves the status of the Elasticsearch cluster for a given sales channel.
     *
     * * @Route("api/_action/sisi/elasticsearch/getStatus", name="api.action.getStatus", methods={"POST"} )
     * @param Request $request The request object containing the sales channel ID.
     * @return JsonResponse A JSON response containing the cluster status.
     */
    public function getStatus(Request $request): JsonResponse
    {
        $salesChannelId = $request->get('salesChannelId');
        $cluster = $this->indexService->getCluster($salesChannelId);

        return new JsonResponse($cluster);
    }
}