<?php

namespace Sisi\Search\Commands;

use Shopware\Core\System\SystemConfig\SystemConfigService;
use Sisi\Search\Components\CliStyle;
use Sisi\Search\Service\TextService;
use Sisi\Search\Util\CliLogger;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * This command is used to start the indexing process for the Sisi Search plugin.
 * It allows to index products in steps, optimizing memory usage.
 * It accepts several arguments to control the indexing process, such as shop, limit, time, update, language, and languageID.
 * @package Sisi\Search\Commands
 *
 * @SuppressWarnings(PHPMD)
 */
#[AsCommand(name: 'sisi-Search-Index:start')]
class Command_SisiSearchIndex extends Command
{
    protected SystemConfigService $systemConfigService;

    protected static $defaultName = 'sisi-Search-Index:start';

    /**
     * Constructor of the class.
     *
     * @param SystemConfigService $config The system configuration service.
     */
    public function __construct(SystemConfigService $config)
    {
        parent::__construct();
        $this->systemConfigService = $config;
    }

    /**
     * Configures the command by defining the arguments.
     */
    protected function configure(): void
    {
        $this->addArgument('shop', InputArgument::OPTIONAL, 'Shop Channel');
        $this->addArgument('shopID', InputArgument::OPTIONAL, 'Shop Channel Id');
        $this->addArgument('limit', InputArgument::OPTIONAL, 'You can add the limit');
        $this->addArgument('main', InputArgument::OPTIONAL, 'Only add main Products in the Index');


        $this->addArgument(
            'time',
            InputArgument::OPTIONAL,
            'Enter the days which will be used for the update index process'
        );
        $this->addArgument(
            'update',
            InputArgument::OPTIONAL,
            'Update the index with update="1"'
        );

        $this->addArgument(
            'language',
            InputArgument::OPTIONAL,
            'This parameter is necessary when you want use not the default language'
        );
        $this->addArgument(
            'languageID',
            InputArgument::OPTIONAL,
            'This parameter is necessary when you want use not the default language and you know the language id'
        );

        $this->addArgument(
            'stop',
            InputArgument::OPTIONAL
        );


        $this->setDescription('Execute the index Memory optimized');
    }

    /**
     * Executes the command to start the indexing process.
     *
     * @param InputInterface $input The input interface.
     * @param OutputInterface $output The output interface.
     *
     * @return int 0 if everything went fine, or an error code.
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        CliLogger::setCliStyle(new CliStyle($input, $output));
        $output->writeln('Sisi Index Command indexer');
        $arguments = $input->getArguments();
        $texthaendler = new TextService();
        $parameter = $texthaendler->stripOption($arguments);
        $config = $this->systemConfigService->get("SisiSearch.config");
        $extendCommand = "";
        $phpPad = "php ";
        $strUpdate = true;
        $strTime = true;

        // ---- Set default limit if not provided
        if (!array_key_exists('limit', $arguments)) {
            $parameter['limit'] = 1000;
        }
        if (empty($parameter['limit'])) {
            $parameter['limit'] = 1000;
        }

        // ---- Build extend command for update parameter
        if (array_key_exists('update', $parameter)) {
            $extendCommand .= " update=" . $parameter['update'];
            $strUpdate = false;
        }

        // ---- Build extend command for language parameter
        if (array_key_exists('language', $parameter) && !array_key_exists('languageID', $parameter)) {
            $extendCommand .= ' language="' . $parameter['language'] . '"';
        }

        // ---- Build extend command for languageID parameter
        if (array_key_exists('languageID', $parameter)) {
            $extendCommand .= ' languageID="' . $parameter['languageID'] . '"';
        }

        // ---- Build extend command for time parameter
        if (array_key_exists('time', $parameter)) {
            $extendCommand .= ' time="' . $parameter['time'] . '"';
            $strTime = false;
        }

        $shop = $input->getArgument('shop');
        $shop = str_replace("shop=", "", $shop);
        $phpPad = $this->getPhpPfad($config);
        $maincommand = $phpPad . 'bin/console sisi-Produkt-Stepindex:start shop="' . $shop . '" limit=' . $parameter['limit'] . $extendCommand;
        $str = true;
        $index = 0;

        // ---- Loop to execute the indexing command in steps
        while ($str) {
            $extendCommand = "";
            if ($index > 0) {
                $extendCommand = " offset=" . $index;
                if ($strUpdate) {
                    $extendCommand .= " update=3";
                }
                if ($strTime) {
                    $extendCommand .= " time=1";
                }
            }
            $extendCommand .= " stop=" .  ($index + $parameter['limit']);
            $command = $maincommand . $extendCommand;
            echo $command . "\n";
            $output = shell_exec($command);
            var_dump($output);
            $pos = strpos($output, "Next 0");
            if ($pos !== false) {
                $str = false;
            }
            $pos = strpos($output, "The index process is finish");
            if ($pos === false) {
                $str = false;
            }
            $index = $index + $parameter['limit'];
        }
        return 0;
    }

    /**
     * Retrieves the PHP path from the configuration.
     *
     * @param array $config The configuration array.
     *
     * @return string The PHP path.
     */
    private function getPhpPfad(array $config): string
    {
        $phpPad = 'php ';
        if (array_key_exists('phpPad', $config)) {
            if (!empty($config['phpPad'])) {
                $phpPad = $config['phpPad'] . ' ';
            }
        }
        return $phpPad;
    }
}