<?php

namespace Sisi\Search\Commands;

use Shopware\Core\Content\Product\AbstractPropertyGroupSorter;
use Shopware\Core\System\SalesChannel\Context\AbstractSalesChannelContextFactory;
use Sisi\Search\Components\CliStyle;
use Sisi\Search\Service\StartService;
use Sisi\Search\ServicesInterfaces\InterfaceSearchCategorieService;
use Sisi\Search\Util\CliLogger;
use Symfony\Bridge\Monolog\Logger;
use Sisi\Search\ESindexing\ProductDataIndexer;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Shopware\Core\Framework\Context;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Doctrine\DBAL\Connection;
use Shopware\Elasticsearch\Framework\ElasticsearchHelper;
use Shopware\Core\Checkout\Cart\Price\QuantityPriceCalculator;
use Sisi\Search\Service\TextService;
use Symfony\Component\ErrorHandler\Debug;
use Topdata\TopdataQueueHelperSW6\Util\UtilDebug;

/**
 * This class represents a Symfony command that triggers a step-by-step indexing process for products.
 * It allows for indexing products in batches, optionally filtering by shop, time, or language.
 * @package Sisi\Search\Commands
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Command_ProductStepIndexer extends Command
{
    /**
     * @var string The default name of the command.
     */
    protected static $defaultName = 'sisi-Produkt-Stepindex:start';

    /**
     * @var Connection The database connection.
     */
    protected $connection;

    /**
     * @var Context The context.
     */
    protected $context;

    /**
     * @var SystemConfigService The system configuration service.
     */
    protected $systemConfigService;

    /**
     * @var ElasticsearchHelper The Elasticsearch helper.
     */
    protected $helper;

    /**
     * @var ContainerInterface The container.
     */
    protected $container;

    /**
     * @var productDataIndexer The product data indexer.
     */
    protected $productDataIndexer;

    /**
     * @var QuantityPriceCalculator The quantity price calculator.
     */
    private $priceCalculator;

    /**
     * @var  AbstractSalesChannelContextFactory The sales channel context factory.
     */
    private $salesChannelContextFactory;

    /**
     * @var Logger The logging service.
     */
    private $logger;

    /**
     * @var InterfaceSearchCategorieService
     */
    protected $searchCategorieService;

    /**
     * @var AbstractPropertyGroupSorter
     */
    protected $propertyGroupSorter;

    /**
     * Constructor for the Command_ProductStepIndexer.
     *
     * @param Connection $connection The database connection.
     * @param ContainerInterface $container The container.
     * @param SystemConfigService $systemConfigService The system configuration service.
     * @param ElasticsearchHelper $helper The Elasticsearch helper.
     * @param ProductDataIndexer $productDataIndexer The product data indexer.
     * @param QuantityPriceCalculator $priceCalculator The quantity price calculator.
     * @param AbstractSalesChannelContextFactory $salesChannelContextFactory The sales channel context factory.
     * @param Logger $loggingService The logging service.
     * @param InterfaceSearchCategorieService $searchCategorieService
     * @param AbstractPropertyGroupSorter $propertyGroupSorter
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Connection                         $connection,
        ContainerInterface                 $container,
        SystemConfigService                $systemConfigService,
        ElasticsearchHelper                $helper,
        ProductDataIndexer                 $productDataIndexer,
        QuantityPriceCalculator            $priceCalculator,
        AbstractSalesChannelContextFactory $salesChannelContextFactory,
        Logger                             $loggingService,
        InterfaceSearchCategorieService    $searchCategorieService,
        AbstractPropertyGroupSorter        $propertyGroupSorter
    ) {
        parent::__construct();
        $this->connection = $connection;
        $this->container = $container;
        $this->systemConfigService = $systemConfigService;
        $this->helper = $helper;
        $this->productDataIndexer = $productDataIndexer;
        $this->priceCalculator = $priceCalculator;
        $this->salesChannelContextFactory = $salesChannelContextFactory;
        $this->logger = $loggingService;
        $this->searchCategorieService = $searchCategorieService;
        $this->propertyGroupSorter = $propertyGroupSorter;
    }

    /**
     * Configures the command by defining the available arguments.
     */
    protected function configure(): void
    {
        $this->addArgument('shop', InputArgument::OPTIONAL, 'Shop Channel Name');
        $this->addArgument('shopID', InputArgument::OPTIONAL, 'Shop Channel Id');
        $this->addArgument('limit', InputArgument::OPTIONAL, 'You can add the limit');
        $this->addArgument('main', InputArgument::OPTIONAL, 'Only add main Products in the Index');


        $this->addArgument(
            'time',
            InputArgument::OPTIONAL,
            'Enter the days which will be used for the update index process'
        );
        $this->addArgument(
            'update',
            InputArgument::OPTIONAL,
            'Update the index with update="1"'
        );

        $this->addArgument(
            'language',
            InputArgument::OPTIONAL,
            'This parameter is necessary when you want use not the default language'
        );
        $this->addArgument(
            'languageID',
            InputArgument::OPTIONAL,
            'This parameter is necessary when you want use not the default language and you know the language id'
        );
        $this->addArgument(
            'offset',
            InputArgument::OPTIONAL
        );
    }

    /**
     * ==== MAIN ====
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        CliLogger::setCliStyle(new CliStyle($input, $output));

        $output->writeln('Step indexer started now please wait..');
        Debug::enable();
        $startService = new StartService();
        $arguments = $input->getArguments();
        $texthaendler = new TextService();
        $parameter = $texthaendler->stripOption($arguments);

        // ---- Set default limit if not provided
        if (!array_key_exists('limit', $arguments)) {
            $parameter['limit'] = 1000;
        }
        if (empty($parameter['limit'])) {
            $parameter['limit'] = 1000;
        }

        $parameter['shop'] = $input->getArgument('shop');
        $parameter['propertyGroupSorter'] = $this->propertyGroupSorter; // wtf?

        // ---- Debug output
        dump($parameter, $this->systemConfigService->get("SisiSearch.config"));

        // ---- Start the indexing process
        $startService->startTheIndexing(
            $this->systemConfigService,
            $this->productDataIndexer,
            $this->connection,
            $this->container,
            $this->priceCalculator,
            $this->salesChannelContextFactory,
            $this->logger,
            $parameter,
            $output,
            $this->searchCategorieService
        );

        return 0;
    }
}