<?php

namespace Sisi\Search\Commands;

use Doctrine\DBAL\Connection;
use Sisi\Search\Service\CategorieIndexService;
use Sisi\Search\Service\TextService;
use Sisi\Search\ServicesInterfaces\InterfaceSearchCategorieService;
use Symfony\Bridge\Monolog\Logger;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Component\Console\Input\InputArgument;
use Sisi\Search\Service\DeleteService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Command to trigger the indexing of categories in Elasticsearch.
 * This command allows specifying a shop, shop ID, and language for indexing.
 * It also provides an option to delete old indexes.
 */
class CategorieIndexCommand extends Command
{
    protected static $defaultName = 'sisi-Produkt-index:categorie';


    protected SystemConfigService $config;


    /**
     * @var Connection
     */
    protected $connection;


    /**
     * @var Logger
     */
    private $loggingService;


    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var InterfaceSearchCategorieService
     */
    protected $searchCategorieService;


    /**
     * Constructor for the CategorieIndexCommand.
     *
     * @param SystemConfigService $config System configuration service.
     * @param Connection $connection Database connection.
     * @param Logger $logger Logging service.
     * @param ContainerInterface $container Dependency injection container.
     * @param InterfaceSearchCategorieService $searchCategorieService Category search service interface.
     */
    public function __construct(
        SystemConfigService             $config,
        Connection                      $connection,
        Logger                          $logger,
        ContainerInterface              $container,
        InterfaceSearchCategorieService $searchCategorieService
    ) {
        parent::__construct();
        $this->config = $config;
        $this->connection = $connection;
        $this->loggingService = $logger;
        $this->container = $container;
        $this->searchCategorieService = $searchCategorieService;
    }


    /**
     * Configures the command with arguments for shop, shop ID, index deletion, language, and language ID.
     */
    protected function configure(): void
    {
        $this->addArgument('shop', InputArgument::OPTIONAL, 'shop Channel');
        $this->addArgument('shopID', InputArgument::OPTIONAL, 'shop Channel id');
        $this->addArgument(
            'all',
            InputArgument::OPTIONAL,
            'Delete all Indexes without the last Indexes. Add the nummber what no want to delete'
        );
        $this->addArgument(
            'language',
            InputArgument::OPTIONAL,
            'With this parameters you only delete indexing from this language'
        );
        $this->addArgument(
            'languageID',
            InputArgument::OPTIONAL,
            'This parameter is necessary when you want use not the default language and you know the language id'
        );
        $this->setDescription('Delete full Elastcsearch indexes for keep space on the ES server');
    }

    /**
     * Executes the command to start the category indexing process.
     *
     * @param InputInterface $input Input from the console.
     * @param OutputInterface $output Output to the console.
     * @return int|null
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        // ---- Initialize the CategorieIndexService
        $categorieIndexService = new  CategorieIndexService();
        $options = $input->getArguments();

        // ---- Initialize the TextService
        $textService = new TextService();

        // ---- Strip options and get parameters
        $paramters = $textService->stripOption2($options);
        $paramters['shop'] = $input->getArgument('shop');

        // ---- Start the indexing process
        $categorieIndexService->startIndex($this->container, $paramters, $this->connection, $this->config, $output, $this->loggingService, $this->searchCategorieService);
        return 0;
    }
}