import template from './sw-order-create-base.html.twig';

/**
 * @sw-package checkout
 */

const { Store, Utils, Data, Service, Mixin } = Shopware;
const { Criteria } = Data;
const { get, format, array } = Utils;

// eslint-disable-next-line sw-deprecation-rules/private-feature-declarations
export default {
    template,

    emits: ['error'],

    mixins: [
        Mixin.getByName('notification'),
    ],

    data() {
        return {
            isLoading: false,
            isLoadingDetail: false,
            address: {
                data: null,
            },
            showAddressModal: false,
            addAddressModalTitle: null,
            editAddressModalTitle: null,
            promotionError: null,
            showPromotionModal: false,
            disabledAutoPromotionChecked: false,
        };
    },

    computed: {
        cartErrors() {
            return Store.get('swOrder').cartErrors;
        },

        customerRepository() {
            return Service('repositoryFactory').create('customer');
        },

        customerAddressRepository() {
            return Service('repositoryFactory').create('customer_address');
        },

        currencyRepository() {
            return Service('repositoryFactory').create('currency');
        },

        customerAddressCriteria() {
            const criteria = new Criteria(1, 25);

            criteria.addAssociation('salutation');
            criteria.addAssociation('country');
            criteria.addAssociation('countryState');

            return criteria;
        },

        defaultCriteria() {
            const criteria = new Criteria(1, 25);
            criteria
                .addAssociation('addresses')
                .addAssociation('group')
                .addAssociation('salutation')
                .addAssociation('salesChannel')
                .addAssociation('lastPaymentMethod')
                .addAssociation('defaultBillingAddress.country')
                .addAssociation('defaultBillingAddress.countryState')
                .addAssociation('defaultBillingAddress.salutation')
                .addAssociation('defaultShippingAddress.country')
                .addAssociation('defaultShippingAddress.countryState')
                .addAssociation('defaultShippingAddress.salutation')
                .addAssociation('tags');

            return criteria;
        },

        orderDate() {
            const today = new Date();
            return format.date(today);
        },

        customer() {
            return Store.get('swOrder').customer;
        },

        salesChannelId() {
            return this.customer?.salesChannelId ?? '';
        },

        isCustomerActive() {
            return Store.get('swOrder').isCustomerActive;
        },

        cart() {
            return Store.get('swOrder').cart;
        },

        cartLineItems() {
            return this.cart.lineItems;
        },

        cartAutomaticPromotionItems() {
            return this.cartLineItems.filter((item) => item.type === 'promotion' && item.payload.code === '');
        },

        cartPrice() {
            return this.cart.price;
        },

        currency() {
            return Store.get('swOrder').context.currency;
        },

        cartDelivery() {
            return get(this.cart, 'deliveries[0]', null);
        },

        promotionCodeTags: {
            get() {
                return Store.get('swOrder').promotionCodes;
            },

            set(promotionCodeTags) {
                Store.get('swOrder').setPromotionCodes(promotionCodeTags);
            },
        },

        cartDeliveryDiscounts() {
            return array.slice(this.cart.deliveries, 1) || [];
        },

        filteredCalculatedTaxes() {
            if (!this.cartPrice || !this.cartPrice.calculatedTaxes) {
                return [];
            }

            return this.sortByTaxRate(this.cartPrice.calculatedTaxes).filter((price) => price.tax !== 0);
        },

        promotionCodeLineItems() {
            return this.cartLineItems.filter((item) => item.type === 'promotion' && get(item, 'payload.code'));
        },

        hasLineItem() {
            return this.cartLineItems.filter((item) => item.hasOwnProperty('id')).length > 0;
        },

        shippingCostsDetail() {
            if (!this.cartDelivery) {
                return null;
            }

            const calcTaxes = this.sortByTaxRate(this.cartDelivery.shippingCosts.calculatedTaxes);
            const decorateCalcTaxes = calcTaxes.map((item) => {
                return this.$tc(
                    'sw-order.createBase.shippingCostsTax',
                    {
                        taxRate: item.taxRate,
                        tax: format.currency(item.tax, this.currency.isoCode),
                    },
                    0,
                );
            });

            return `${this.$tc('sw-order.createBase.tax')}<br>${decorateCalcTaxes.join('<br>')}`;
        },

        disabledAutoPromotionVisibility: {
            get() {
                return this.disabledAutoPromotionChecked;
            },
            set(visibility) {
                this.switchAutomaticPromotions(visibility);
            },
        },

        taxStatus() {
            return get(this.cart, 'price.taxStatus', '');
        },

        displayRounded() {
            if (!this.cartPrice) {
                return false;
            }
            return this.cartPrice.rawTotal !== this.cartPrice.totalPrice;
        },

        orderTotal() {
            if (!this.cartPrice) {
                return 0;
            }

            if (this.displayRounded) {
                return this.cartPrice.rawTotal;
            }

            return this.cartPrice.totalPrice;
        },

        currencyFilter() {
            return Shopware.Filter.getByName('currency');
        },
    },

    watch: {
        cart: {
            deep: true,
            handler: 'updatePromotionList',
        },

        promotionCodeTags: {
            handler: 'handlePromotionCodeTags',
        },

        cartErrors: {
            handler(newValue) {
                if (!newValue || newValue.length === 0) {
                    return;
                }

                Object.values(newValue).forEach((value) => {
                    switch (value.level) {
                        case 0: {
                            this.createNotificationSuccess({
                                message: value.message,
                            });
                            break;
                        }

                        case 10: {
                            this.createNotificationWarning({
                                message: value.message,
                            });
                            break;
                        }

                        default: {
                            this.createNotificationError({
                                message: value.message,
                            });
                            break;
                        }
                    }
                });
            },
        },
    },

    created() {
        this.createdComponent();
    },

    methods: {
        createdComponent() {
            const { customer } = this.$route.params;

            if (!customer) {
                return;
            }

            Store.get('swOrder').setCustomer(customer);
            this.onSelectExistingCustomer(customer.id);
        },

        async createCart(salesChannelId) {
            await Store.get('swOrder').createCart({ salesChannelId });
        },

        async loadCart() {
            if (!this.cart.token || this.cart.lineItems.length === 0) return;
            this.updateLoading(true);

            Store.get('swOrder')
                .getCart({
                    salesChannelId: this.customer.salesChannelId,
                    contextToken: this.cart.token,
                })
                .finally(() => this.updateLoading(false));
        },

        async onSelectExistingCustomer(customerId) {
            this.isLoadingDetail = true;

            try {
                const customer = await this.customerRepository.get(customerId, Shopware.Context.api, this.defaultCriteria);

                if (!this.cart.token) {
                    await this.createCart(customer.salesChannelId);
                }

                this.setCustomer(customer);
                this.setCurrency(customer);

                await this.updateCustomerContext();
            } catch {
                this.createNotificationError({
                    message: this.$tc('sw-order.create.messageSwitchCustomerError'),
                });
            } finally {
                this.isLoadingDetail = false;
            }
        },

        async updateCustomerContext() {
            await Store.get('swOrder').updateCustomerContext({
                customerId: this.customer.id,
                salesChannelId: this.customer.salesChannelId,
                contextToken: this.cart.token,
            });
        },

        setCustomer(customer) {
            Store.get('swOrder').selectExistingCustomer({ customer });
        },

        setCurrency(customer) {
            this.currencyRepository.get(customer.salesChannel.currencyId).then((currency) => {
                Store.get('swOrder').setCurrency(currency);
            });
        },

        onEditBillingAddress() {
            const contextId = 'billingAddressId';
            const contextDataKey = 'billingAddress';
            const contextDataDefaultId = 'defaultBillingAddressId';
            const data = this.customer[contextDataKey] ? this.customer[contextDataKey] : this.customer.defaultBillingAddress;

            this.addAddressModalTitle = this.$tc('sw-order.addressSelection.modalTitleAddBillingAddress');
            this.editAddressModalTitle = this.$tc('sw-order.addressSelection.modalTitleEditBillingAddress');
            this.address = {
                contextId,
                contextDataKey,
                contextDataDefaultId,
                data,
            };
            this.showAddressModal = true;
        },

        onEditShippingAddress() {
            const contextId = 'shippingAddressId';
            const contextDataKey = 'shippingAddress';
            const contextDataDefaultId = 'defaultShippingAddressId';
            const data = this.customer[contextDataKey]
                ? this.customer[contextDataKey]
                : this.customer.defaultShippingAddress;

            this.addAddressModalTitle = this.$tc('sw-order.addressSelection.modalTitleAddShippingAddress');
            this.editAddressModalTitle = this.$tc('sw-order.addressSelection.modalTitleEditShippingAddress');
            this.address = {
                contextId,
                contextDataKey,
                contextDataDefaultId,
                data,
            };
            this.showAddressModal = true;
        },

        setCustomerAddress({ contextId, data }) {
            this.customer[contextId] = data.id;
            const availableCustomerAddresses = [
                {
                    id: this.customer.billingAddressId,
                    dataKey: 'billingAddress',
                },
                {
                    id: this.customer.shippingAddressId,
                    dataKey: 'shippingAddress',
                },
                {
                    id: this.customer.defaultBillingAddressId,
                    dataKey: 'defaultBillingAddress',
                },
                {
                    id: this.customer.defaultShippingAddressId,
                    dataKey: 'defaultShippingAddress',
                },
            ];

            this.customerAddressRepository
                .get(data.id, Shopware.Context.api, this.customerAddressCriteria)
                .then((updatedAddress) => {
                    availableCustomerAddresses.forEach((customerAddress) => {
                        if (customerAddress.id === data.id) {
                            this.customer[customerAddress.dataKey] = updatedAddress;
                        }
                    });

                    this.setCustomer(this.customer);
                });
        },

        closeModal() {
            this.showAddressModal = false;
            this.address.data = null;
        },

        save() {
            this.closeModal();
        },

        onSaveItem(item) {
            this.updateLoading(true);

            Store.get('swOrder')
                .saveLineItem({
                    salesChannelId: this.customer.salesChannelId,
                    contextToken: this.cart.token,
                    item,
                })
                .finally(() => this.updateLoading(false));
        },

        onRemoveItems(lineItemKeys) {
            this.updateLoading(true);

            Store.get('swOrder')
                .removeLineItems({
                    salesChannelId: this.customer.salesChannelId,
                    contextToken: this.cart.token,
                    lineItemKeys: lineItemKeys,
                })
                .then(() => {
                    // Remove promotion code tag if corresponding line item removed
                    lineItemKeys.forEach((key) => {
                        const removedTag = this.promotionCodeTags.find((tag) => tag.discountId === key);
                        if (removedTag) {
                            this.promotionCodeTags = this.promotionCodeTags.filter((item) => {
                                return item.discountId !== removedTag.discountId;
                            });
                        }
                    });
                })
                .finally(() => this.updateLoading(false));
        },

        updateLoading(loadingValue) {
            this.isLoading = loadingValue;
        },

        sortByTaxRate(price) {
            return price.sort((prev, current) => {
                return prev.taxRate - current.taxRate;
            });
        },

        onSubmitCode(code) {
            this.updateLoading(true);

            Store.get('swOrder')
                .addPromotionCode({
                    salesChannelId: this.customer.salesChannelId,
                    contextToken: this.cart.token,
                    code,
                })
                .finally(() => this.updateLoading(false));
        },

        onRemoveExistingCode(item) {
            if (item.isInvalid) {
                this.promotionCodeTags = this.promotionCodeTags.filter((tag) => tag.code !== item.code);
            } else {
                this.onRemoveItems([item.discountId]);
            }
        },

        updatePromotionList() {
            // Update data and isInvalid flag for each item in promotionCodeTags
            this.promotionCodeTags = this.promotionCodeTags.map((tag) => {
                const matchedItem = this.promotionCodeLineItems.find((lineItem) => lineItem.payload.code === tag.code);

                if (matchedItem) {
                    return { ...matchedItem.payload, isInvalid: false };
                }

                return { ...tag, isInvalid: true };
            });

            // Add new items from promotionCodeLineItems which promotionCodeTags doesn't contain
            this.promotionCodeLineItems.forEach((lineItem) => {
                const matchedItem = this.promotionCodeTags.find((tag) => tag.code === lineItem.payload.code);

                if (!matchedItem) {
                    this.promotionCodeTags = [
                        ...this.promotionCodeTags,
                        { ...lineItem.payload, isInvalid: false },
                    ];
                }
            });
        },

        handlePromotionCodeTags(newValue, oldValue) {
            this.promotionError = null;

            if (newValue.length < oldValue.length) {
                return;
            }

            const promotionCodeLength = this.promotionCodeTags.length;
            const latestTag = this.promotionCodeTags[promotionCodeLength - 1];

            if (newValue.length > oldValue.length) {
                this.onSubmitCode(latestTag.code);
            }

            if (promotionCodeLength > 0 && latestTag.isInvalid) {
                this.promotionError = {
                    detail: this.$tc('sw-order.createBase.textInvalidPromotionCode'),
                };
            }
        },

        onShippingChargeEdited() {
            this.updateLoading(true);

            Store.get('swOrder')
                .modifyShippingCosts({
                    salesChannelId: this.customer.salesChannelId,
                    contextToken: this.cart.token,
                    shippingCosts: this.cartDelivery.shippingCosts,
                })
                .catch((error) => {
                    this.$emit('error', error);
                })
                .finally(() => {
                    this.updateLoading(false);
                });
        },

        switchAutomaticPromotions(visibility) {
            this.disabledAutoPromotionChecked = visibility;
            this.showPromotionModal = visibility;
            if (!this.showPromotionModal) {
                this.enableAutomaticPromotions();
            }
        },

        enableAutomaticPromotions() {
            this.updateLoading(true);
            const additionalParams = {
                salesChannelId: this.customer.salesChannelId,
            };
            Service('cartStoreService')
                .enableAutomaticPromotions(this.cart.token, additionalParams)
                .then(() => {
                    this.loadCart();
                });
        },

        onClosePromotionModal() {
            this.showPromotionModal = false;
            this.disabledAutoPromotionChecked = false;
        },

        onSavePromotionModal() {
            this.showPromotionModal = false;
            this.disabledAutoPromotionChecked = true;

            this.loadCart();
        },

        onShippingChargeUpdated(amount) {
            const positiveAmount = Math.abs(amount);
            this.cartDelivery.shippingCosts.unitPrice = positiveAmount;
            this.cartDelivery.shippingCosts.totalPrice = positiveAmount;
        },
    },
};
