<?php declare(strict_types=1);

namespace Shopware\Storefront\Theme;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Media\Exception\DuplicatedMediaFileNameException;
use Shopware\Core\Content\Media\File\FileNameProvider;
use Shopware\Core\Content\Media\File\FileSaver;
use Shopware\Core\Content\Media\File\MediaFile;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Write\Validation\RestrictDeleteViolationException;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\Language\LanguageEntity;
use Shopware\Storefront\Theme\StorefrontPluginConfiguration\StorefrontPluginConfiguration;
use Shopware\Storefront\Theme\StorefrontPluginConfiguration\StorefrontPluginConfigurationCollection;
use function GuzzleHttp\Psr7\mimetype_from_filename;

class ThemeLifecycleService
{
    private StorefrontPluginRegistryInterface $pluginRegistry;

    private EntityRepositoryInterface $themeRepository;

    private EntityRepositoryInterface $mediaRepository;

    private EntityRepositoryInterface $mediaFolderRepository;

    private EntityRepositoryInterface $themeMediaRepository;

    private FileSaver $fileSaver;

    private ThemeFileImporterInterface $themeFileImporter;

    private FileNameProvider $fileNameProvider;

    private EntityRepositoryInterface $languageRepository;

    private EntityRepositoryInterface $themeChildRepository;

    private Connection $connection;

    /**
     * @internal
     */
    public function __construct(
        StorefrontPluginRegistryInterface $pluginRegistry,
        EntityRepositoryInterface $themeRepository,
        EntityRepositoryInterface $mediaRepository,
        EntityRepositoryInterface $mediaFolderRepository,
        EntityRepositoryInterface $themeMediaRepository,
        FileSaver $fileSaver,
        FileNameProvider $fileNameProvider,
        ThemeFileImporterInterface $themeFileImporter,
        EntityRepositoryInterface $languageRepository,
        EntityRepositoryInterface $themeChildRepository,
        Connection $connection
    ) {
        $this->pluginRegistry = $pluginRegistry;
        $this->themeRepository = $themeRepository;
        $this->mediaRepository = $mediaRepository;
        $this->mediaFolderRepository = $mediaFolderRepository;
        $this->themeMediaRepository = $themeMediaRepository;
        $this->fileSaver = $fileSaver;
        $this->fileNameProvider = $fileNameProvider;
        $this->themeFileImporter = $themeFileImporter;
        $this->languageRepository = $languageRepository;
        $this->themeChildRepository = $themeChildRepository;
        $this->connection = $connection;
    }

    public function refreshThemes(
        Context $context,
        ?StorefrontPluginConfigurationCollection $configurationCollection = null
    ): void {
        if ($configurationCollection === null) {
            $configurationCollection = $this->pluginRegistry->getConfigurations()->getThemes();
        }

        // iterate over all theme configs in the filesystem (plugins/bundles)
        foreach ($configurationCollection as $config) {
            $this->refreshTheme($config, $context);
        }
    }

    public function refreshTheme(StorefrontPluginConfiguration $configuration, Context $context): void
    {
        $themeData['name'] = $configuration->getName();
        $themeData['technicalName'] = $configuration->getTechnicalName();
        $themeData['author'] = $configuration->getAuthor();

        // refresh theme after deleting media
        $theme = $this->getThemeByTechnicalName($configuration->getTechnicalName(), $context);

        // check if theme config already exists in the database
        if ($theme) {
            $themeData['id'] = $theme->getId();
        } else {
            $themeData['active'] = true;
        }

        $themeData['translations'] = $this->getTranslationsConfiguration($configuration, $context);

        $updatedData = $this->updateMediaInConfiguration($theme, $configuration, $context);

        $themeData = array_merge($themeData, $updatedData);

        if (!empty($configuration->getConfigInheritance())) {
            $themeData = $this->addParentTheme($configuration, $themeData, $context);
        }

        $writtenEvent = $this->themeRepository->upsert([$themeData], $context);

        if (!isset($themeData['id']) || empty($themeData['id'])) {
            $themeData['id'] = current($writtenEvent->getPrimaryKeys(ThemeDefinition::ENTITY_NAME));
        }

        $this->themeRepository->upsert([$themeData], $context);

        $parentThemes = $this->getParentThemes($configuration, $themeData['id']);
        $parentCriteria = new Criteria();
        $parentCriteria->addFilter(new EqualsFilter('childId', $themeData['id']));
        $toDeleteIds = $this->themeChildRepository->searchIds($parentCriteria, $context);
        $this->themeChildRepository->delete($toDeleteIds->getIds(), $context);
        $this->themeChildRepository->upsert($parentThemes, $context);
    }

    public function removeTheme(string $technicalName, Context $context): void
    {
        $criteria = new Criteria();
        $criteria->addAssociation('dependentThemes');
        $criteria->addFilter(new EqualsFilter('technicalName', $technicalName));

        /** @var ThemeEntity|null $theme */
        $theme = $this->themeRepository->search($criteria, $context)->first();

        if ($theme === null) {
            return;
        }

        $dependentThemes = $theme->getDependentThemes() ?? new ThemeCollection();
        $ids = array_merge(array_values($dependentThemes->getIds()), [$theme->getId()]);

        $this->removeOldMedia($technicalName, $context);
        $this->themeRepository->delete(array_map(function (string $id) {
            return ['id' => $id];
        }, $ids), $context);
    }

    private function getThemeByTechnicalName(string $technicalName, Context $context): ?ThemeEntity
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('technicalName', $technicalName));

        return $this->themeRepository->search($criteria, $context)->first();
    }

    private function createMediaStruct(string $path, string $mediaId, ?string $themeFolderId): ?array
    {
        $path = $this->themeFileImporter->getRealPath($path);

        if (!$this->fileExists($path)) {
            return null;
        }

        $pathinfo = pathinfo($path);

        return [
            'basename' => $pathinfo['filename'],
            'media' => ['id' => $mediaId, 'mediaFolderId' => $themeFolderId],
            'mediaFile' => new MediaFile(
                $path,
                mimetype_from_filename($pathinfo['basename']),
                $pathinfo['extension'] ?? '',
                filesize($path)
            ),
        ];
    }

    private function getMediaDefaultFolderId(Context $context): ?string
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('media_folder.defaultFolder.entity', 'theme'));
        $criteria->addAssociation('defaultFolder');
        $criteria->setLimit(1);
        $defaultFolder = $this->mediaFolderRepository->search($criteria, $context);
        $defaultFolderId = null;
        if ($defaultFolder->count() === 1) {
            $defaultFolderId = $defaultFolder->first()->getId();
        }

        return $defaultFolderId;
    }

    private function getTranslationsConfiguration(StorefrontPluginConfiguration $configuration, Context $context): array
    {
        $systemLanguageLocale = $this->getSystemLanguageLocale($context);

        $labelTranslations = $this->getLabelsFromConfig($configuration->getThemeConfig());
        $translations = $this->mapTranslations($labelTranslations, 'labels', $systemLanguageLocale);

        $helpTextTranslations = $this->getHelpTextsFromConfig($configuration->getThemeConfig());

        return array_merge_recursive(
            $translations,
            $this->mapTranslations($helpTextTranslations, 'helpTexts', $systemLanguageLocale)
        );
    }

    private function getLabelsFromConfig(array $config): array
    {
        $translations = [];
        if (\array_key_exists('blocks', $config)) {
            $translations = array_merge_recursive($translations, $this->extractLabels('blocks', $config['blocks']));
        }

        if (\array_key_exists('sections', $config)) {
            $translations = array_merge_recursive($translations, $this->extractLabels('sections', $config['sections']));
        }

        if (\array_key_exists('tabs', $config)) {
            $translations = array_merge_recursive($translations, $this->extractLabels('tabs', $config['tabs']));
        }

        if (\array_key_exists('fields', $config)) {
            $translations = array_merge_recursive($translations, $this->extractLabels('fields', $config['fields']));
        }

        return $translations;
    }

    private function extractLabels(string $prefix, array $data): array
    {
        $labels = [];
        foreach ($data as $key => $item) {
            if (\array_key_exists('label', $item)) {
                foreach ($item['label'] as $locale => $label) {
                    $labels[$locale][$prefix . '.' . $key] = $label;
                }
            }
        }

        return $labels;
    }

    private function getHelpTextsFromConfig(array $config): array
    {
        $translations = [];

        if (\array_key_exists('fields', $config)) {
            $translations = array_merge_recursive($translations, $this->extractHelpTexts('fields', $config['fields']));
        }

        return $translations;
    }

    private function extractHelpTexts(string $prefix, array $data): array
    {
        $helpTexts = [];
        foreach ($data as $key => $item) {
            if (!isset($item['helpText'])) {
                continue;
            }

            foreach ($item['helpText'] as $locale => $label) {
                $helpTexts[$locale][$prefix . '.' . $key] = $label;
            }
        }

        return $helpTexts;
    }

    private function fileExists(string $path): bool
    {
        return $this->themeFileImporter->fileExists($path);
    }

    private function removeOldMedia(string $technicalName, Context $context): void
    {
        $theme = $this->getThemeByTechnicalName($technicalName, $context);

        if (!$theme) {
            return;
        }

        // find all assigned media files
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('media.themeMedia.id', $theme->getId()));
        $result = $this->mediaRepository->searchIds($criteria, $context);

        // delete theme media association
        $themeMediaData = [];
        foreach ($result->getIds() as $id) {
            $themeMediaData[] = ['themeId' => $theme->getId(), 'mediaId' => $id];
        }

        if (empty($themeMediaData)) {
            return;
        }

        // remove associations between theme and media first
        $this->themeMediaRepository->delete($themeMediaData, $context);

        // delete media associated with theme
        foreach ($themeMediaData as $item) {
            try {
                $this->mediaRepository->delete([['id' => $item['mediaId']]], $context);
            } catch (RestrictDeleteViolationException $e) {
                // don't delete files that are associated with other entities.
                // This files will be recreated using the file name strategy for duplicated filenames.
            }
        }
    }

    private function updateMediaInConfiguration(
        ?ThemeEntity $theme,
        StorefrontPluginConfiguration $pluginConfiguration,
        Context $context
    ): array {
        $media = [];
        $themeData = [];
        $themeFolderId = $this->getMediaDefaultFolderId($context);

        if ($pluginConfiguration->getPreviewMedia()) {
            $mediaId = Uuid::randomHex();
            $path = $pluginConfiguration->getPreviewMedia();

            $mediaItem = $this->createMediaStruct($path, $mediaId, $themeFolderId);

            if ($mediaItem) {
                $themeData['previewMediaId'] = $mediaId;
                $media[$path] = $mediaItem;
            }

            // if preview was not deleted because it is not created from theme use current preview id
            if ($theme && $theme->getPreviewMediaId() !== null) {
                $themeData['previewMediaId'] = $theme->getPreviewMediaId();
            }
        }

        $baseConfig = $pluginConfiguration->getThemeConfig();

        if (\array_key_exists('fields', $baseConfig)) {
            foreach ($baseConfig['fields'] as $key => $field) {
                if (!\array_key_exists('type', $field) || $field['type'] !== 'media') {
                    continue;
                }

                $path = $pluginConfiguration->getBasePath() . \DIRECTORY_SEPARATOR . $field['value'];

                if (!\array_key_exists($path, $media)) {
                    $mediaId = Uuid::randomHex();
                    $mediaItem = $this->createMediaStruct($path, $mediaId, $themeFolderId);

                    if (!$mediaItem) {
                        continue;
                    }

                    $media[$path] = $mediaItem;

                    // replace media path with media ids
                    $baseConfig['fields'][$key]['value'] = $mediaId;
                } else {
                    $baseConfig['fields'][$key]['value'] = $media[$path]['media']['id'];
                }
            }
            $themeData['baseConfig'] = $baseConfig;
        }

        $mediaIds = [];

        if (!empty($media)) {
            $mediaIds = array_column($media, 'media');

            $this->mediaRepository->create($mediaIds, $context);

            foreach ($media as $item) {
                try {
                    $this->fileSaver->persistFileToMedia($item['mediaFile'], $item['basename'], $item['media']['id'], $context);
                } catch (DuplicatedMediaFileNameException $e) {
                    $newFileName = $this->fileNameProvider->provide(
                        $item['basename'],
                        $item['mediaFile']->getFileExtension(),
                        null,
                        $context
                    );
                    $this->fileSaver->persistFileToMedia($item['mediaFile'], $newFileName, $item['media']['id'], $context);
                }
            }
        }

        $themeData['media'] = $mediaIds;

        return $themeData;
    }

    private function getSystemLanguageLocale(Context $context): string
    {
        $criteria = new Criteria();
        $criteria->addAssociation('translationCode');
        $criteria->addFilter(new EqualsFilter('id', Defaults::LANGUAGE_SYSTEM));

        /** @var LanguageEntity $language */
        $language = $this->languageRepository->search($criteria, $context)->first();

        return $language->getTranslationCode()->getCode();
    }

    private function mapTranslations(array $translations, string $property, string $systemLanguageLocale): array
    {
        $result = [];
        $containsSystemLanguage = false;
        foreach ($translations as $locale => $translation) {
            if ($locale === $systemLanguageLocale) {
                $containsSystemLanguage = true;
            }
            $result[$locale] = [$property => $translation];
        }

        if (!$containsSystemLanguage && \count($translations) > 0) {
            $translation = array_shift($translations);
            if (\array_key_exists('en-GB', $translations)) {
                $translation = $translations['en-GB'];
            }
            $result[$systemLanguageLocale] = [$property => $translation];
        }

        return $result;
    }

    private function addParentTheme(StorefrontPluginConfiguration $configuration, array $themeData, Context $context): array
    {
        $lastNotSameTheme = null;
        foreach (array_reverse($configuration->getConfigInheritance()) as $themeName) {
            if (
                $themeName === '@' . StorefrontPluginRegistry::BASE_THEME_NAME
                || $themeName === '@' . $themeData['technicalName']
            ) {
                continue;
            }
            $lastNotSameTheme = str_replace('@', '', $themeName);
        }

        if ($lastNotSameTheme !== null) {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('technicalName', $lastNotSameTheme));
            /** @var ThemeEntity|null $parentTheme */
            $parentTheme = $this->themeRepository->search($criteria, $context)->first();
            if ($parentTheme) {
                $themeData['parentThemeId'] = $parentTheme->getId();
            }
        }

        return $themeData;
    }

    private function getParentThemes(StorefrontPluginConfiguration $config, string $id): array
    {
        $allThemeConfigs = $this->pluginRegistry->getConfigurations()->getThemes();

        $allThemes = $this->getAllThemesPlain();

        $parentThemeConfigs = $allThemeConfigs->filter(
            fn (StorefrontPluginConfiguration $parentConfig) => $this->isDependentTheme($parentConfig, $config)
        );

        $technicalNames = $parentThemeConfigs->map(
            fn (StorefrontPluginConfiguration $theme) => $theme->getTechnicalName()
        );

        $parentThemes = array_filter(
            $allThemes,
            fn (array $theme) => \in_array($theme['technicalName'] ?? '', $technicalNames, true)
        );

        $updateParents = [];
        /** @var array $parentTheme */
        foreach ($parentThemes as $parentTheme) {
            $updateParents[] = [
                'parentId' => $parentTheme['parentThemeId'] ?? '',
                'childId' => $id,
            ];
        }

        return $updateParents;
    }

    private function getAllThemesPlain(): array
    {
        return $this->connection->fetchAllAssociative(
            'SELECT theme.technical_name as technicalName, LOWER(HEX(theme.id)) as parentThemeId FROM theme'
        );
    }

    private function isDependentTheme(
        StorefrontPluginConfiguration $parentConfig,
        StorefrontPluginConfiguration $currentThemeConfig
    ): bool {
        return $currentThemeConfig->getTechnicalName() !== $parentConfig->getTechnicalName()
            && \in_array('@' . $parentConfig->getTechnicalName(), $currentThemeConfig->getStyleFiles()->getFilepaths(), true)
        ;
    }
}
