<?php declare(strict_types=1);

namespace Shopware\Core\System\CustomEntity\Schema;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Schema\AbstractSchemaManager;
use Doctrine\DBAL\Schema\Comparator;
use Doctrine\DBAL\Schema\Schema;
use Symfony\Component\Lock\LockFactory;

/**
 * @internal
 * @phpstan-import-type CustomEntityField from SchemaUpdater
 */
class CustomEntitySchemaUpdater
{
    private const COMMENT = 'custom-entity-element';

    private Connection $connection;

    private LockFactory $lockFactory;

    private SchemaUpdater $schemaUpdater;

    public function __construct(Connection $connection, LockFactory $lockFactory, SchemaUpdater $schemaUpdater)
    {
        $this->connection = $connection;
        $this->lockFactory = $lockFactory;
        $this->schemaUpdater = $schemaUpdater;
    }

    public function update(): void
    {
        $this->lock(function (): void {
            /** @var list<array{name: string, fields: string}> $tables */
            $tables = $this->connection->fetchAllAssociative('SELECT name, fields FROM custom_entity');

            $schema = $this->getSchemaManager()->createSchema();

            $this->cleanup($schema);

            $this->schemaUpdater->applyCustomEntities($schema, $tables);

            $this->applyNewSchema($schema);
        });
    }

    private function lock(\Closure $closure): void
    {
        $lock = $this->lockFactory->createLock('custom-entity::schema-update', 30);

        if ($lock->acquire(true)) {
            $closure();

            $lock->release();
        }
    }

    private function applyNewSchema(Schema $update): void
    {
        $baseSchema = $this->getSchemaManager()->createSchema();
        $queries = (new Comparator())
            ->compare($baseSchema, $update)
            ->toSql($this->getPlatform());

        foreach ($queries as $query) {
            try {
                $this->connection->executeStatement($query);
            } catch (Exception $e) {
                // there seems to be a timing issue in sql when dropping a foreign key which relates to an index.
                // Sometimes the index exists no more when doctrine tries to drop it after dropping the foreign key.
                if (!\str_contains($e->getMessage(), 'An exception occurred while executing \'DROP INDEX IDX_')) {
                    throw $e;
                }
            }
        }
    }

    private function getSchemaManager(): AbstractSchemaManager
    {
        $manager = $this->connection->getSchemaManager();
        if (!$manager instanceof AbstractSchemaManager) {
            throw new \RuntimeException('The schema manager could not be found.');
        }

        return $manager;
    }

    private function getPlatform(): AbstractPlatform
    {
        $platform = $this->connection->getDatabasePlatform();
        if (!$platform instanceof AbstractPlatform) {
            throw new \RuntimeException('Database platform can not be detected');
        }

        return $platform;
    }

    private function cleanup(Schema $schema): void
    {
        foreach ($schema->getTables() as $table) {
            if ($table->getComment() === self::COMMENT) {
                $schema->dropTable($table->getName());

                continue;
            }

            foreach ($table->getForeignKeys() as $foreignKey) {
                if (\str_starts_with($foreignKey->getName(), 'fk_ce_')) {
                    $table->removeForeignKey($foreignKey->getName());
                }
            }

            foreach ($table->getColumns() as $column) {
                if ($column->getComment() === self::COMMENT) {
                    $table->dropColumn($column->getName());
                }
            }
        }
    }
}
