<?php declare(strict_types=1);

namespace Shopware\Core\Checkout\Test\Document\DocumentType;

use Doctrine\DBAL\Connection;
use PHPUnit\Framework\TestCase;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\CartBehavior;
use Shopware\Core\Checkout\Cart\Exception\InvalidPayloadException;
use Shopware\Core\Checkout\Cart\Exception\InvalidQuantityException;
use Shopware\Core\Checkout\Cart\Exception\MixedLineItemTypeException;
use Shopware\Core\Checkout\Cart\Order\OrderPersister;
use Shopware\Core\Checkout\Cart\Processor;
use Shopware\Core\Checkout\Document\DocumentConfiguration;
use Shopware\Core\Checkout\Document\DocumentConfigurationFactory;
use Shopware\Core\Checkout\Document\DocumentGenerator\InvoiceGenerator;
use Shopware\Core\Checkout\Document\DocumentGenerator\StornoGenerator;
use Shopware\Core\Checkout\Document\FileGenerator\PdfGenerator;
use Shopware\Core\Checkout\Document\GeneratedDocument;
use Shopware\Core\Checkout\Order\OrderEntity;
use Shopware\Core\Checkout\Test\Cart\Common\TrueRule;
use Shopware\Core\Checkout\Test\Payment\Handler\V630\SyncTestPaymentHandler;
use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Shopware\Core\Content\Product\Cart\ProductLineItemFactory;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\TaxFreeConfig;
use Shopware\Core\Framework\Feature;
use Shopware\Core\Framework\Rule\Collector\RuleConditionRegistry;
use Shopware\Core\Framework\Test\TestCaseBase\CountryAddToSalesChannelTestBehaviour;
use Shopware\Core\Framework\Test\TestCaseBase\IntegrationTestBehaviour;
use Shopware\Core\Framework\Test\TestCaseBase\TaxAddToSalesChannelTestBehaviour;
use Shopware\Core\Framework\Test\TestCaseHelper\ReflectionHelper;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\Currency\CurrencyFormatter;
use Shopware\Core\System\DeliveryTime\DeliveryTimeEntity;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextFactory;
use Shopware\Core\System\SalesChannel\Context\SalesChannelContextService;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Test\TestDefaults;

/**
 * @internal
 */
class InvoiceServiceTest extends TestCase
{
    use IntegrationTestBehaviour;
    use TaxAddToSalesChannelTestBehaviour;
    use CountryAddToSalesChannelTestBehaviour;

    private SalesChannelContext $salesChannelContext;

    private Context $context;

    private Connection $connection;

    private CurrencyFormatter $currencyFormatter;

    protected function setUp(): void
    {
        Feature::skipTestIfActive('v6.5.0.0', $this);

        parent::setUp();

        $this->context = Context::createDefaultContext();
        $this->connection = $this->getContainer()->get(Connection::class);
        $this->currencyFormatter = $this->getContainer()->get(CurrencyFormatter::class);

        $priceRuleId = Uuid::randomHex();
        $customerId = $this->createCustomer();
        $shippingMethodId = $this->createShippingMethod($priceRuleId);
        $paymentMethodId = $this->createPaymentMethod($priceRuleId);

        $this->addCountriesToSalesChannel();

        $this->salesChannelContext = $this->getContainer()->get(SalesChannelContextFactory::class)->create(
            Uuid::randomHex(),
            TestDefaults::SALES_CHANNEL,
            [
                SalesChannelContextService::CUSTOMER_ID => $customerId,
                SalesChannelContextService::SHIPPING_METHOD_ID => $shippingMethodId,
                SalesChannelContextService::PAYMENT_METHOD_ID => $paymentMethodId,
            ]
        );

        $this->salesChannelContext->setRuleIds([$priceRuleId]);
    }

    public function testGenerateWithDifferentTaxes(): void
    {
        $invoiceService = $this->getContainer()->get(InvoiceGenerator::class);
        $pdfGenerator = $this->getContainer()->get(PdfGenerator::class);

        $possibleTaxes = [7, 19, 22];
        //generates one line item for each tax
        $cart = $this->generateDemoCart($possibleTaxes);
        $orderId = $this->persistCart($cart);
        /** @var OrderEntity $order */
        $order = $this->getOrderById($orderId);

        $documentConfiguration = DocumentConfigurationFactory::mergeConfiguration(
            new DocumentConfiguration(),
            [
                'displayLineItems' => true,
                'itemsPerPage' => 10,
                'displayFooter' => true,
                'displayHeader' => true,
            ]
        );
        $context = Context::createDefaultContext();

        $processedTemplate = $invoiceService->generate(
            $order,
            $documentConfiguration,
            $context
        );

        static::assertNotNull($order->getCurrency());
        static::assertNotNull($lineItems = $order->getLineItems());
        static::assertNotNull($firstLineItem = $lineItems->first());
        static::assertNotNull($lastLineItem = $lineItems->last());
        static::assertStringContainsString('<html>', $processedTemplate);
        static::assertStringContainsString('</html>', $processedTemplate);
        static::assertStringContainsString($firstLineItem->getLabel(), $processedTemplate);
        static::assertStringContainsString($lastLineItem->getLabel(), $processedTemplate);
        static::assertStringContainsString(
            $this->currencyFormatter->formatCurrencyByLanguage(
                $order->getAmountTotal(),
                $order->getCurrency()->getIsoCode(),
                $context->getLanguageId(),
                $context
            ),
            $processedTemplate
        );
        foreach ($possibleTaxes as $possibleTax) {
            static::assertStringContainsString(
                sprintf('plus %d%% VAT', $possibleTax),
                $processedTemplate
            );
        }

        $generatedDocument = new GeneratedDocument();
        $generatedDocument->setHtml($processedTemplate);

        $generatorOutput = $pdfGenerator->generate($generatedDocument);
        static::assertNotEmpty($generatorOutput);

        $finfo = new \finfo(\FILEINFO_MIME_TYPE);
        static::assertEquals('application/pdf', $finfo->buffer($generatorOutput));

        $deLanguageId = $this->getDeDeLanguageId();
        $order->setLanguageId($deLanguageId);

        $processedTemplate = $invoiceService->generate(
            $order,
            $documentConfiguration,
            $context
        );

        static::assertNotNull($order->getCurrency());
        static::assertStringContainsString(
            preg_replace('/\xc2\xa0/', ' ', $this->currencyFormatter->formatCurrencyByLanguage(
                $order->getAmountTotal(),
                $order->getCurrency()->getIsoCode(),
                $deLanguageId,
                $context
            )) ?: '',
            preg_replace('/\xc2\xa0/', ' ', $processedTemplate) ?: ''
        );
    }

    public function testGenerateStornoWithDifferentTaxes(): void
    {
        $stornoGenerator = $this->getContainer()->get(StornoGenerator::class);
        $pdfGenerator = $this->getContainer()->get(PdfGenerator::class);

        $possibleTaxes = [7, 19, 22];
        //generates one line item for each tax
        $cart = $this->generateDemoCart($possibleTaxes);
        $orderId = $this->persistCart($cart);
        /** @var OrderEntity $order */
        $order = $this->getOrderById($orderId);

        $documentConfiguration = DocumentConfigurationFactory::mergeConfiguration(
            new DocumentConfiguration(),
            [
                'displayLineItems' => true,
                'itemsPerPage' => 10,
                'displayFooter' => true,
                'displayHeader' => true,
            ]
        );
        $context = Context::createDefaultContext();

        $processedTemplate = $stornoGenerator->generate(
            $order,
            $documentConfiguration,
            $context
        );

        static::assertNotNull($order->getCurrency());
        static::assertNotNull($lineItems = $order->getLineItems());
        static::assertNotNull($firstLineItem = $lineItems->first());
        static::assertNotNull($lastLineItem = $lineItems->last());
        static::assertLessThanOrEqual(0, $order->getPrice()->getTotalPrice());
        static::assertLessThanOrEqual(0, $order->getPrice()->getRawTotal());
        static::assertStringContainsString('<html>', $processedTemplate);
        static::assertStringContainsString('</html>', $processedTemplate);
        static::assertStringContainsString($firstLineItem->getLabel(), $processedTemplate);
        static::assertStringContainsString($lastLineItem->getLabel(), $processedTemplate);
        static::assertStringContainsString(
            $this->currencyFormatter->formatCurrencyByLanguage(
                $order->getAmountTotal(),
                $order->getCurrency()->getIsoCode(),
                $context->getLanguageId(),
                $context
            ),
            $processedTemplate
        );
        foreach ($possibleTaxes as $possibleTax) {
            static::assertStringContainsString(
                sprintf('plus %d%% VAT', $possibleTax),
                $processedTemplate
            );
        }

        $generatedDocument = new GeneratedDocument();
        $generatedDocument->setHtml($processedTemplate);

        $generatorOutput = $pdfGenerator->generate($generatedDocument);
        static::assertNotEmpty($generatorOutput);

        $finfo = new \finfo(\FILEINFO_MIME_TYPE);
        static::assertEquals('application/pdf', $finfo->buffer($generatorOutput));
    }

    public function testGenerateWithShippingAddress(): void
    {
        $invoiceService = $this->getContainer()->get(InvoiceGenerator::class);

        $possibleTaxes = [7, 19, 22];
        //generates one line item for each tax
        $cart = $this->generateDemoCart($possibleTaxes);
        $orderId = $this->persistCart($cart);
        $order = $this->getOrderById($orderId);
        static::assertNotNull($order);
        static::assertNotNull($deliveries = $order->getDeliveries());
        static::assertNotNull($shippingAddresses = $deliveries->getShippingAddress());
        static::assertNotNull($countries = $shippingAddresses->getCountries());
        static::assertNotNull($country = $countries->first());
        $country->setCompanyTax(new TaxFreeConfig(true, Defaults::CURRENCY, 0));
        $companyPhone = '123123123';

        static::assertNotNull($order->getOrderCustomer());

        $documentConfiguration = DocumentConfigurationFactory::mergeConfiguration(
            new DocumentConfiguration(),
            [
                'displayLineItems' => true,
                'itemsPerPage' => 10,
                'displayFooter' => true,
                'displayHeader' => true,
                'executiveDirector' => true,
                'displayDivergentDeliveryAddress' => true,
                'companyPhone' => $companyPhone,
                'intraCommunityDelivery' => true,
                'displayAdditionalNoteDelivery' => true,
                'deliveryCountries' => [$country->getId()],
            ]
        );

        $context = Context::createDefaultContext();

        $processedTemplate = $invoiceService->generate(
            $order,
            $documentConfiguration,
            $context
        );

        static::assertNotNull($orderDeliveries = $order->getDeliveries());
        static::assertNotNull($shippingAddresses = $orderDeliveries->getShippingAddress());
        static::assertNotNull($shippingAddress = $shippingAddresses->first());
        static::assertStringContainsString('Shipping address', $processedTemplate);
        static::assertStringContainsString($shippingAddress->getStreet(), $processedTemplate);
        static::assertStringContainsString($shippingAddress->getCity(), $processedTemplate);
        static::assertStringContainsString($shippingAddress->getFirstName(), $processedTemplate);
        static::assertStringContainsString($shippingAddress->getLastName(), $processedTemplate);
        static::assertStringContainsString($shippingAddress->getZipcode(), $processedTemplate);
        static::assertStringContainsString('Intra-community delivery (EU)', $processedTemplate);
        static::assertStringContainsString($companyPhone, $processedTemplate);
    }

    /**
     * @dataProvider invoiceGenerateVatIdProvider
     */
    public function testGenerateWithVatIdsOfCustomer(\Closure $vatIdClosure, \Closure $assertClosure): void
    {
        $invoiceService = $this->getContainer()->get(InvoiceGenerator::class);

        $possibleTaxes = [7, 19, 22];
        $cart = $this->generateDemoCart($possibleTaxes);
        $orderId = $this->persistCart($cart);
        /** @var OrderEntity $order */
        $order = $this->getOrderById($orderId);

        $vatId = $vatIdClosure($order);

        static::assertNotNull($deliveries = $order->getDeliveries());
        static::assertNotNull($shippingAddress = $deliveries->getShippingAddress());
        static::assertNotNull($countries = $shippingAddress->getCountries());
        static::assertNotNull($country = $countries->first());
        $documentConfiguration = DocumentConfigurationFactory::mergeConfiguration(
            new DocumentConfiguration(),
            [
                'displayLineItems' => true,
                'itemsPerPage' => 10,
                'displayFooter' => true,
                'displayHeader' => true,
                'executiveDirector' => true,
                'displayDivergentDeliveryAddress' => true,
                'intraCommunityDelivery' => true,
                'displayAdditionalNoteDelivery' => true,
                'deliveryCountries' => [$country->getId()],
            ]
        );

        $context = Context::createDefaultContext();

        $processedTemplate = $invoiceService->generate(
            $order,
            $documentConfiguration,
            $context
        );

        $assertClosure($processedTemplate, $vatId);
    }

    public function testGenerateWhenUncheckedDisplayLineItems(): void
    {
        $invoiceService = $this->getContainer()->get(InvoiceGenerator::class);

        $possibleTaxes = [7, 19, 22];
        $cart = $this->generateDemoCart($possibleTaxes);
        $orderId = $this->persistCart($cart);
        /** @var OrderEntity $order */
        $order = $this->getOrderById($orderId);

        static::assertNotNull($order->getLineItems());
        $lineItem = $order->getLineItems()->first();

        $documentConfiguration = DocumentConfigurationFactory::mergeConfiguration(
            new DocumentConfiguration(),
            [
                'displayLineItems' => false,
                'itemsPerPage' => 10,
                'displayFooter' => true,
                'displayHeader' => true,
                'executiveDirector' => true,
                'displayDivergentDeliveryAddress' => true,
                'intraCommunityDelivery' => true,
                'displayAdditionalNoteDelivery' => true,
            ]
        );

        $context = Context::createDefaultContext();

        $processedTemplate = $invoiceService->generate(
            $order,
            $documentConfiguration,
            $context
        );

        static::assertNotNull($lineItem);
        static::assertNotNull($lineItem->getLabel());
        static::assertNotNull($lineItem->getPayload());
        static::assertNotNull($lineItem->getPayload()['productNumber']);

        static::assertStringNotContainsString($lineItem->getLabel(), $processedTemplate);
        static::assertStringNotContainsString($lineItem->getPayload()['productNumber'], $processedTemplate);
    }

    public function invoiceGenerateVatIdProvider(): \Generator
    {
        $vatId = 'VAT-123123';

        yield 'Generate an invoice with the customer has no VAT' => [
            function (OrderEntity $order) use ($vatId): string {
                return $vatId;
            },
            function ($processedTemplate, $vatId): void {
                static::assertStringNotContainsString("VAT Reg.No: ${vatId}", $processedTemplate);
            },
        ];

        yield 'Generate an invoice with the customer has to VAT with disabled company tax and a customer country is not part of the European Union' => [
            function (OrderEntity $order) use ($vatId): string {
                static::assertNotNull($orderCustomer = $order->getOrderCustomer());
                static::assertNotNull($customer = $orderCustomer->getCustomer());
                $customer->setVatIds([$vatId]);

                return $vatId;
            },
            function ($processedTemplate, $vatId): void {
                static::assertStringNotContainsString("VAT Reg.No: ${vatId}", $processedTemplate);
            },
        ];

        yield 'Generate an invoice with the customer has to VAT with enabled company tax and a customer country is not part of the European Union' => [
            function (OrderEntity $order) use ($vatId): string {
                static::assertNotNull($orderCustomer = $order->getOrderCustomer());
                static::assertNotNull($customer = $orderCustomer->getCustomer());
                $customer->setVatIds([$vatId]);

                static::assertNotNull($addresses = $order->getAddresses());
                static::assertNotNull($billingAddress = $addresses->get($order->getBillingAddressId()));
                static::assertNotNull($country = $billingAddress->getCountry());
                $country->getCompanyTax()->setEnabled(true);

                static::assertNotNull($addresses = $order->getAddresses());
                static::assertNotNull($billingAddress = $addresses->get($order->getBillingAddressId()));
                static::assertNotNull($country = $billingAddress->getCountry());
                $country->setId(Uuid::randomBytes());

                return $vatId;
            },
            function ($processedTemplate, $vatId): void {
                static::assertStringNotContainsString("VAT Reg.No: ${vatId}", $processedTemplate);
            },
        ];

        yield 'Generate an invoice with the customer has to VAT with company tax, and a customer country is part of the European Union' => [
            function (OrderEntity $order) use ($vatId): string {
                static::assertNotNull($orderCustomer = $order->getOrderCustomer());
                static::assertNotNull($customer = $orderCustomer->getCustomer());
                $customer->setVatIds([$vatId]);

                static::assertNotNull($addresses = $order->getAddresses());
                static::assertNotNull($billingAddress = $addresses->get($order->getBillingAddressId()));
                static::assertNotNull($country = $billingAddress->getCountry());
                $country->getCompanyTax()->setEnabled(true);

                return $vatId;
            },
            function ($processedTemplate, $vatId): void {
                static::assertStringContainsString("VAT Reg.No: ${vatId}", $processedTemplate);
            },
        ];
    }

    /**
     * @param array<int, int> $taxes
     *
     * @throws InvalidPayloadException
     * @throws InvalidQuantityException
     * @throws MixedLineItemTypeException
     * @throws \Exception
     */
    private function generateDemoCart(array $taxes): Cart
    {
        $cart = new Cart('A', 'a-b-c');

        $keywords = ['awesome', 'epic', 'high quality'];

        $products = [];

        $factory = new ProductLineItemFactory();

        foreach ($taxes as $tax) {
            $id = Uuid::randomHex();

            $price = random_int(100, 200000) / 100.0;

            shuffle($keywords);
            $name = ucfirst(implode(' ', $keywords) . ' product');

            $products[] = [
                'id' => $id,
                'name' => $name,
                'price' => [
                    ['currencyId' => Defaults::CURRENCY, 'gross' => $price, 'net' => $price, 'linked' => false],
                ],
                'productNumber' => Uuid::randomHex(),
                'manufacturer' => ['id' => $id, 'name' => 'test'],
                'tax' => ['id' => $id, 'taxRate' => $tax, 'name' => 'test'],
                'stock' => 10,
                'active' => true,
                'visibilities' => [
                    ['salesChannelId' => TestDefaults::SALES_CHANNEL, 'visibility' => ProductVisibilityDefinition::VISIBILITY_ALL],
                ],
            ];

            $cart->add($factory->create($id));
            $this->addTaxDataToSalesChannel($this->salesChannelContext, end($products)['tax']);
        }

        $this->getContainer()->get('product.repository')
            ->create($products, Context::createDefaultContext());

        $cart = $this->getContainer()->get(Processor::class)->process($cart, $this->salesChannelContext, new CartBehavior());

        return $cart;
    }

    private function persistCart(Cart $cart): string
    {
        $orderId = $this->getContainer()->get(OrderPersister::class)->persist($cart, $this->salesChannelContext);

        return $orderId;
    }

    private function createCustomer(): string
    {
        $customerId = Uuid::randomHex();
        $addressId = Uuid::randomHex();

        $customer = [
            'id' => $customerId,
            'number' => '1337',
            'salutationId' => $this->getValidSalutationId(),
            'firstName' => 'Max',
            'lastName' => 'Mustermann',
            'customerNumber' => '1337',
            'email' => Uuid::randomHex() . '@example.com',
            'password' => 'shopware',
            'defaultPaymentMethodId' => $this->getDefaultPaymentMethod(),
            'groupId' => TestDefaults::FALLBACK_CUSTOMER_GROUP,
            'salesChannelId' => TestDefaults::SALES_CHANNEL,
            'defaultBillingAddressId' => $addressId,
            'defaultShippingAddressId' => $addressId,
            'addresses' => [
                [
                    'id' => $addressId,
                    'customerId' => $customerId,
                    'countryId' => $this->getValidCountryId(),
                    'salutationId' => $this->getValidSalutationId(),
                    'firstName' => 'Max',
                    'lastName' => 'Mustermann',
                    'street' => 'Ebbinghoff 10',
                    'zipcode' => '48624',
                    'city' => 'Schöppingen',
                ],
            ],
        ];

        $this->getContainer()->get('customer.repository')->upsert([$customer], $this->context);

        return $customerId;
    }

    private function createShippingMethod(string $priceRuleId): string
    {
        $shippingMethodId = Uuid::randomHex();
        $repository = $this->getContainer()->get('shipping_method.repository');

        $ruleRegistry = $this->getContainer()->get(RuleConditionRegistry::class);
        $prop = ReflectionHelper::getProperty(RuleConditionRegistry::class, 'rules');
        $prop->setValue($ruleRegistry, array_merge($prop->getValue($ruleRegistry), ['true' => new TrueRule()]));

        $data = [
            'id' => $shippingMethodId,
            'type' => 0,
            'name' => 'test shipping method',
            'bindShippingfree' => false,
            'active' => true,
            'prices' => [
                [
                    'name' => 'Std',
                    'currencyPrice' => [
                        [
                            'currencyId' => Defaults::CURRENCY,
                            'net' => 10.00,
                            'gross' => 10.00,
                            'linked' => false,
                        ],
                    ],
                    'currencyId' => Defaults::CURRENCY,
                    'calculation' => 1,
                    'quantityStart' => 1,
                ],
            ],
            'deliveryTime' => $this->createDeliveryTimeData(),
            'availabilityRule' => [
                'id' => $priceRuleId,
                'name' => 'true',
                'priority' => 1,
                'conditions' => [
                    [
                        'type' => (new TrueRule())->getName(),
                    ],
                ],
            ],
        ];

        $repository->create([$data], $this->context);

        return $shippingMethodId;
    }

    /**
     * @return array<string, string|int>
     */
    private function createDeliveryTimeData(): array
    {
        return [
            'id' => Uuid::randomHex(),
            'name' => 'test',
            'min' => 1,
            'max' => 90,
            'unit' => DeliveryTimeEntity::DELIVERY_TIME_DAY,
        ];
    }

    private function createPaymentMethod(string $ruleId): string
    {
        $paymentMethodId = Uuid::randomHex();
        $repository = $this->getContainer()->get('payment_method.repository');

        $ruleRegistry = $this->getContainer()->get(RuleConditionRegistry::class);
        $prop = ReflectionHelper::getProperty(RuleConditionRegistry::class, 'rules');
        $prop->setValue($ruleRegistry, array_merge($prop->getValue($ruleRegistry), ['true' => new TrueRule()]));

        $data = [
            'id' => $paymentMethodId,
            'handlerIdentifier' => SyncTestPaymentHandler::class,
            'name' => 'Payment',
            'active' => true,
            'position' => 0,
            'availabilityRules' => [
                [
                    'id' => $ruleId,
                    'name' => 'true',
                    'priority' => 0,
                    'conditions' => [
                        [
                            'type' => 'true',
                        ],
                    ],
                ],
            ],
            'salesChannels' => [
                [
                    'id' => TestDefaults::SALES_CHANNEL,
                ],
            ],
        ];

        $repository->create([$data], $this->context);

        return $paymentMethodId;
    }

    /**
     * @throws InconsistentCriteriaIdsException
     *
     * @return mixed|null
     */
    private function getOrderById(string $orderId)
    {
        $criteria = (new Criteria([$orderId]))
            ->addAssociation('lineItems')
            ->addAssociation('currency')
            ->addAssociation('language.locale')
            ->addAssociation('transactions')
            ->addAssociation('deliveries.shippingOrderAddress.country')
            ->addAssociation('orderCustomer.customer')
            ->addAssociation('addresses.country');

        $order = $this->getContainer()->get('order.repository')
            ->search($criteria, $this->context)
            ->get($orderId);

        static::assertNotNull($orderId);

        return $order;
    }

    private function getDefaultPaymentMethod(): ?string
    {
        $id = $this->connection->fetchOne(
            'SELECT `id` FROM `payment_method` WHERE `active` = 1 ORDER BY `position` ASC'
        );

        if (!$id) {
            return null;
        }

        return Uuid::fromBytesToHex($id);
    }
}
