import { shallowMount } from '@vue/test-utils';
import 'src/module/sw-customer/page/sw-customer-create';
import 'src/app/component/base/sw-button';
import 'src/app/component/base/sw-button-process';

const { Context } = Shopware;
const { EntityCollection } = Shopware.Data;

async function createWrapper() {
    return shallowMount(await Shopware.Component.build('sw-customer-create'), {
        stubs: {
            'sw-page': true,
            'sw-card': true,
            'sw-language-switch': true,
            'sw-customer-address-form': true,
            'sw-customer-base-form': true,
            'sw-card-view': true,
            'sw-button': await Shopware.Component.build('sw-button'),
            'sw-button-process': await Shopware.Component.build('sw-button-process'),
            'sw-icon': true
        },
        provide: {
            numberRangeService: {},
            systemConfigApiService: {
                getValues: () => Promise.resolve({ 'core.register.minPasswordLength': 8 })
            },
            customerValidationService: {},
            repositoryFactory: {
                create: (entity) => {
                    if (entity === 'customer') {
                        return {
                            create: () => {
                                return {
                                    id: '63e27affb5804538b5b06cb4e344b130',
                                    addresses: new EntityCollection('/customer_address', 'customer_address', Context.api, null, []),
                                };
                            }
                        };
                    }

                    return {
                        create: () => Promise.resolve()
                    };
                }
            }
        }
    });
}

describe('module/sw-customer/page/sw-customer-create', () => {
    it('should have valid email validation response when no email is given', async () => {
        const wrapper = await createWrapper();
        await wrapper.vm.$nextTick();

        await wrapper.setData({
            customer: {
                id: '1',
                email: null,
                boundSalesChannelId: null
            }
        });

        const response = await wrapper.vm.validateEmail();

        expect(response.isValid).toBe(true);
    });

    it('should have company validation when customer type is commercial', async () => {
        const wrapper = await createWrapper();
        wrapper.vm.createNotificationError = jest.fn();
        wrapper.vm.validateEmail = jest.fn().mockImplementation(() => Promise.resolve({ isValid: true }));
        const notificationMock = wrapper.vm.createNotificationError;

        await wrapper.setData({
            customer: {
                id: '1',
                email: 'user@domain.com',
                accountType: 'business',
                password: 'shopware',
            },
            address: {
                company: ''
            }
        });
        const saveButton = wrapper.find('.sw-customer-create__save-action');
        await saveButton.trigger('click');
        await wrapper.vm.$nextTick();

        expect(notificationMock).toBeCalledTimes(1);
        expect(notificationMock).toHaveBeenCalledWith({
            message: 'sw-customer.error.COMPANY_IS_REQUIRED'
        });

        wrapper.vm.createNotificationError.mockRestore();
    });

    it('should show an error notification when entered password is invalid', async () => {
        const wrapper = await createWrapper();
        wrapper.vm.createNotificationError = jest.fn();
        const notificationMock = wrapper.vm.createNotificationError;

        await wrapper.setData({
            customer: {
                id: '1',
                email: 'test@shopware.com',
                accountType: 'business',
                password: '1234567',
            },
        });

        const saveButton = wrapper.find('.sw-customer-create__save-action');
        await saveButton.trigger('click');
        await wrapper.vm.$nextTick();

        expect(notificationMock).toBeCalledTimes(1);
        expect(notificationMock).toHaveBeenCalledWith({
            message: 'sw-customer.detail.notificationPasswordLengthErrorMessage'
        });

        wrapper.vm.createNotificationError.mockRestore();
    });
});
