import axios from 'axios';
import createHTTPClient from 'src/core/factory/http.factory';
import MockAdapter from 'axios-mock-adapter';

Shopware.Application.view.deleteReactive = () => {};

describe('core/factory/http.factory.js', () => {
    let httpClient;
    let mock;

    beforeEach(() => {
        httpClient = createHTTPClient();
        mock = new MockAdapter(httpClient);
    });

    it('should create a HTTP client with response interceptors', async () => {
        expect(Object.getPrototypeOf(httpClient).isPrototypeOf(axios)).toBeTruthy();
    });

    it('should not intercept if store session has not expired', async () => {
        mock.onGet('/store-session-expired').replyOnce(200, {});

        expect(mock.history.get.length).toBe(0);

        await httpClient.get('/store-session-expired');

        expect(mock.history.get.length).toBe(1);
    });

    it.each([
        ['FRAMEWORK__STORE_SESSION_EXPIRED'],
        ['FRAMEWORK__STORE_SHOP_SECRET_INVALID'],
    ])('should intercept and retry if error code matches', async (errorCode) => {
        mock.onGet('/store-route-requiring-auth').replyOnce(403, {
            errors: [{
                code: errorCode,
            }]
        }).onGet('/store-route-requiring-auth').replyOnce(200, {});

        expect(mock.history.get.length).toBe(0);

        await httpClient.get('/store-route-requiring-auth');

        expect(mock.history.get.length).toBe(2);
    });

    it.each([
        ['FRAMEWORK__STORE_SESSION_EXPIRED'],
        ['FRAMEWORK__STORE_SHOP_SECRET_INVALID'],
    ])('should reject the request and reset the counter once the retry limit is hit', async (errorCode) => {
        mock.onGet('/store-route-requiring-auth').reply(403, {
            errors: [{
                code: errorCode,
            }]
        });

        try {
            await httpClient.get('/store-route-requiring-auth');
        } catch (e) {
            expect(e.response.status).toBe(403);
            expect(e.response.data).toEqual({
                errors: [{
                    code: errorCode,
                }]
            });
        }

        expect(mock.history.get.length).toBe(2);
    });

    it.each([
        ['FRAMEWORK__STORE_SESSION_EXPIRED'],
        ['FRAMEWORK__STORE_SHOP_SECRET_INVALID'],
    ])('should treat each request separately', async (errorCode) => {
        mock.onGet('/store-route-requiring-auth').reply(403, {
            errors: [{
                code: errorCode,
            }]
        });

        try {
            await Promise.all([
                httpClient.get('/store-route-requiring-auth'),
                httpClient.get('/store-route-requiring-auth')
            ]);
        } catch (e) {
            expect(e.response.status).toBe(403);
            expect(e.response.data).toEqual({
                errors: [{
                    code: errorCode,
                }]
            });
        }

        expect(mock.history.get.length).toBe(4);
    });
});
